## ----setup, include=FALSE-----------------------------------------
source("_setup.R")

## -----------------------------------------------------------------
library(labestData)
ls("package:labestData")

## ---- eval=FALSE--------------------------------------------------
#  help(BanzattoQd3.2.1, help_type = "html")

## -----------------------------------------------------------------
#-----------------------------------------------------------------------
# Ler a partir do repositório do labestData.

# url <- paste0("https://gitlab.c3sl.ufpr.br/pet-estatistica",
#               "/labestData/raw/devel/data-raw/BanzattoQd3.2.1.txt")
#
# BanzattoQd3.2.1 <-
#     read.table(file = url, sep = "\t", header = TRUE,
#                colClasses = c("factor", "integer", "integer"))

#-----------------------------------------------------------------------
# Análise exploratória.

# Estrutura do objeto.
str(BanzattoQd3.2.1)

# Tabela de frequência para os tratamentos.
xtabs(~trat, data = BanzattoQd3.2.1)

# Dados desemplilhados.
t(unstack(x = BanzattoQd3.2.1, form = pulgoes ~ trat))

# Média e desvio-padrão das observações em cada nível.
aggregate(pulgoes ~ trat,  data = BanzattoQd3.2.1,
          FUN = function(y) {
              c(mean = mean(y), dp = sd(y))
          })

library(lattice)
packageVersion("lattice")

# Diagrama de dispersão.
xyplot(pulgoes ~ reorder(trat, pulgoes), data = BanzattoQd3.2.1,
       xlab = "Produtos para controle de pulgão",
       ylab = "Número de pulgões 36hs após pulverização",
       scales = list(x = list(rot = 90)))


## -----------------------------------------------------------------
#-----------------------------------------------------------------------
# Ajuste do modelo.

m0 <- lm(pulgoes ~ trat, data = BanzattoQd3.2.1)

# Estimativas dos efeitos. Restrição de zerar primeiro nível.
cbind(coef(m0))

# Matriz de contrastes sob a retrição zerar primeiro nível.
K <- cbind(1, contrasts(BanzattoQd3.2.1$trat))
K

# Médias estimadas pelo modelo.
K %*% coef(m0)

# Médias amostrais.
aggregate(pulgoes ~ trat, data = BanzattoQd3.2.1, FUN = mean)

## -----------------------------------------------------------------
m1 <- update(m0, contrasts = list(trat = "contr.SAS"))
coef(m1)

BanzattoQd3.2.1 <- transform(BanzattoQd3.2.1,
                             trat = relevel(trat, ref = "Testemunha"))

levels(BanzattoQd3.2.1$trat)

m0 <- update(m0,  data = BanzattoQd3.2.1)
coef(m0)

## -----------------------------------------------------------------
#-----------------------------------------------------------------------
# Exibe o quarteto fantástico da avaliação dos pressupostos.
par(mfrow = c(2, 2))
plot(m0); layout(1)

## -----------------------------------------------------------------
MASS::boxcox(m0)
abline(v = c(0, 1/3), lty = 2, col = 2)

## -----------------------------------------------------------------
# Atualiza o modelo anterior passando o log na resposta.
m0 <- update(m0, log(.) ~ .)

par(mfrow = c(2, 2))
plot(m0); layout(1)

## -----------------------------------------------------------------
anova(m0)

## -----------------------------------------------------------------
summary(m0)

## ---- include=FALSE-----------------------------------------------
k <- nlevels(BanzattoQd3.2.1$trat) - 1
p <- 1 - (1 - 0.05)^k

## -----------------------------------------------------------------
pred <- data.frame(trat = levels(BanzattoQd3.2.1$trat))
pred <- cbind(pred,
              as.data.frame(predict(m0,
                                    newdata = pred,
                                    interval = "confidence")))
pred$trat <- reorder(pred$trat, pred$fit)
head(pred)

library(latticeExtra)

segplot(trat ~ lwr + upr, centers = fit, data = pred,
        draw = FALSE, horizontal = FALSE,
        xlab = "Produtos para controle de pulgão",
        ylab = "Log do número de pulgões",
        scales = list(x = list(rot = 90)),
        panel = function(x, y, z, centers, ...) {
            panel.segplot(x = x, y = y, z = z, centers = centers, ...)
            panel.text(x = as.numeric(z), y = centers,
                       label = sprintf("%0.2f", centers),
                       srt = 90, cex = 0.8, adj = c(0.5, -0.5))
    })

## ---- eval=FALSE--------------------------------------------------
#  trat <- rep(x = c(levels(BanzattoQd3.2.1$trat), "Novo"),
#              times = 8)
#  
#  # Opção 1: sorteie os níveis para as unidades experimentais ordenadas.
#  data.frame(trat = sample(trat), ue = 1:length(trat))
#  
#  # Opção 2: sorteie as unidades experimentais ordenadas para os níveis.
#  data.frame(trat = trat, ue = sample(1:length(trat)))

## -----------------------------------------------------------------
sessionInfo()

