## ----setup, include=FALSE-----------------------------------------
source("_setup.R")

## -----------------------------------------------------------------
library(labestData)
ls("package:labestData")

## ---- eval=FALSE--------------------------------------------------
#  help(PimentelEg5.2, help_type = "html")

## -----------------------------------------------------------------
#-----------------------------------------------------------------------
# Ler a partir do repositório do labestData.

# url <- paste0("https://gitlab.c3sl.ufpr.br/pet-estatistica",
#               "/labestData/raw/devel/data-raw/PimentelEg5.2.txt")
#
# PimentelEg5.2 <-
#     read.table(file = url, sep = "\t", header = TRUE)

#-----------------------------------------------------------------------
# Análise exploratória.

# Estrutura do objeto.
str(PimentelEg5.2)

# Tabela de frequência para os tratamentos.
xtabs(~variedade + bloco, data = PimentelEg5.2)

# Dados desempilhados.
t(unstack(x = PimentelEg5.2, form = producao ~ variedade))

library(lattice)

# Para uma melhor exibição dos dados, vamos reordenar os níveis
# de cultivar pela média de cada uma delas. Depois vamos ordenar os
# registros no data.frame por bloco/variedade.

# Reordena os níveis.
PimentelEg5.2 <- transform(PimentelEg5.2,
                           variedade = reorder(variedade, producao))

# Reordena os registros.
PimentelEg5.2 <- PimentelEg5.2[
    with(PimentelEg5.2, order(bloco, variedade)), ]

# Diagrama de dispersão.
xyplot(producao ~ variedade, data = PimentelEg5.2,
       groups = bloco, type = "o",
       auto.key = list(corner = c(0.05, 0.95), columns = 1,
                       title = "Bloco", cex.title = 1.1),
       xlab = "Variedades de batatinha (ordenadas)",
       ylab = expression("Produção de batatinha"~(t~ha^{-1})),
       scales = list(x = list(rot = 90)))

## -----------------------------------------------------------------
#-----------------------------------------------------------------------
# Ajuste do modelo.

m0 <- lm(producao ~ bloco + variedade, data = PimentelEg5.2)

# Estimativas dos efeitos. Restrição de zerar primeiro nível.
cbind(coef(m0))

# Aqui tem-se o grupo de coeficientes para cada um dos termos do
# preditor para a média (\mu = 0, \alpha_i = 1, \tau_j = 2).
split(coef(m0), m0$assign)

# Matrizes de contrastes utilizadas, sob a retrição zerar primeiro nível.
contrasts(PimentelEg5.2$bloco)
contrasts(PimentelEg5.2$variedade)

# Médias amostrais.
aggregate(producao ~ variedade, data = PimentelEg5.2, FUN = mean)

## -----------------------------------------------------------------
m1 <- update(m0, contrasts = list(bloco = "contr.SAS"))
split(data.frame(coef(m0), coef(m1), row.names = NULL), m0$assign)

# Médias ajustadas de mínimos quadrados (least squares means).
# L <- doBy::LSmatrix(m1, effect = "variedade")
L <- matrix(c(1, 0.25, 0.25, 0.25, 0, 0, 0, 0, 0, 0, 0,
              1, 0.25, 0.25, 0.25, 1, 0, 0, 0, 0, 0, 0,
              1, 0.25, 0.25, 0.25, 0, 1, 0, 0, 0, 0, 0,
              1, 0.25, 0.25, 0.25, 0, 0, 1, 0, 0, 0, 0,
              1, 0.25, 0.25, 0.25, 0, 0, 0, 1, 0, 0, 0,
              1, 0.25, 0.25, 0.25, 0, 0, 0, 0, 1, 0, 0,
              1, 0.25, 0.25, 0.25, 0, 0, 0, 0, 0, 1, 0,
              1, 0.25, 0.25, 0.25, 0, 0, 0, 0, 0, 0, 1),
            byrow = TRUE,
            nrow = nlevels(PimentelEg5.2$variedade),
            dimnames = list(levels(PimentelEg5.2$variedade), NULL))

L %*% coef(m1)

## -----------------------------------------------------------------
#-----------------------------------------------------------------------
# Exibe o quarteto fantástico da avaliação dos pressupostos.

par(mfrow = c(2, 2))
plot(m0); layout(1)


## -----------------------------------------------------------------
#-----------------------------------------------------------------------
# Teste da aditividade bloco x tratamento.

X <- model.matrix(m0)
beta <- coef(m0)
a <- m0$assign
aditiv <- X[, a == 1] %*% beta[a == 1] * X[, a == 2] %*% beta[a == 2]

anova(update(m0, . ~ . + aditiv))

# Pacote agricolae possuí função para o teste da aditividade.
# agricolae::nonadditivity(y = PimentelEg5.2$prod,
#                          factor1 = PimentelEg5.2$bloco,
#                          factor2 = PimentelEg5.2$variedade,
#                          df = df.residual(m0),
#                          MSerror = deviance(m0)/df.residual(m0))


## -----------------------------------------------------------------
anova(m0)

## -----------------------------------------------------------------
# Predição das médias das variades no bloco I.
pred <- data.frame(variedade = levels(PimentelEg5.2$variedade),
                   bloco = "I")
pred <- cbind(pred,
              as.data.frame(predict(m0,
                                    newdata = pred,
                                    interval = "confidence")))
pred$variedade <- reorder(pred$variedade, pred$fit)
pred

## -----------------------------------------------------------------
suppressMessages(library(multcomp, quietly = TRUE))

# Vetor de pesos para o valor esperado da variedade 1 no bloco I e na
# média dos blocos.
l1 <- matrix(c(1, 0,    0,    0,    0, 0, 0, 0, 0, 0, 0), nrow = 1)
l0 <- matrix(c(1, 0.25, 0.25, 0.25, 0, 0, 0, 0, 0, 0, 0), nrow = 1)

# Os erros padrões também são diferentes, assim como as médias.
summary(glht(m0, linfct = l1))
summary(glht(m0, linfct = l0))

# Erros-padrões obtidos matricialmente.
# sqrt(l1 %*% vcov(m0) %*% t(l1))
# sqrt(l0 %*% vcov(m0) %*% t(l0))

# A prova de que a variância do efeito de bloco não contribui para a
# variância da média quando se usa a média dos blocos.
round(data.frame("bloco I" = t(l1 %*% vcov(m0)),
                 "média" = t(l0 %*% vcov(m0))), 5)

## -----------------------------------------------------------------
# IC *individual* de cobertura 95%.
# ic <- confint(glht(m0, linfct = L), calpha = univariate_calpha())
# ic <- as.data.frame(ic$confint)

# IC *global* de cobertura 95%. CUIDADO, essa função demora muito quando
# o número de níveis é grande.
ic <- confint(glht(m0, linfct = L))
ic <- as.data.frame(ic$confint)

names(ic) <- c("fit", "lwr", "upr")

pred <- cbind(variedade = pred$variedade, ic)
pred

library(latticeExtra)

segplot(variedade ~ lwr + upr, centers = fit, data = pred,
        draw = FALSE, horizontal = FALSE,
        xlab = "Variedades de batatinha",
        ylab = expression("Produção de batatinha"~(t~ha^{-1})),
        scales = list(x = list(rot = 90)),
        panel = function(x, y, z, centers, ...) {
            panel.segplot(x = x, y = y, z = z, centers = centers, ...)
            panel.text(x = as.numeric(z), y = centers,
                       label = sprintf("%0.2f", centers),
                       srt = 90, cex = 0.8, adj = c(0.5, -0.5))
        })

## -----------------------------------------------------------------
replicate(nlevels(PimentelEg5.2$bloco),
          sample(levels(PimentelEg5.2$variedade)))

## -----------------------------------------------------------------
sessionInfo()

