## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----eval=FALSE---------------------------------------------------------------
#  library(devtools)
#  install_git("bonatwagner/mglm4twin")

## ----eval=TRUE, error=FALSE, message=FALSE, warning=FALSE---------------------
library(mglm4twin)
packageVersion("mglm4twin")

require(mvtnorm)

## -----------------------------------------------------------------------------
data(anthro)
head(anthro)

## Standardize
anthro$age <- (anthro$age - mean(anthro$age))/sd(anthro$age)
anthro$weight <- (anthro$weight - mean(anthro$weight))/sd(anthro$weight)
anthro$height <- (anthro$height - mean(anthro$height))/sd(anthro$height)


## -----------------------------------------------------------------------------
form_Wt <- weight ~ age + Group*Twin_pair
form_Ht <- height ~ age + Group*Twin_pair

## -----------------------------------------------------------------------------
## ACE model
biv0 <- list("formE1" = ~ age, "formE2" = ~ age, "formE12" = ~ age,
             "formA1" = ~ age, "formA2" = ~ age, "formA12" = ~ age,
             "formC1" = ~ age, "formC2" = ~ age, "formC12" = ~ age)
Z_biv0 <- mt_twin(N_DZ = 327, N_MZ = 534, n_resp = 2, model = "ACE",
                    formula = biv0, data = anthro)

## Special case of ACE model
biv4 <- list("formE1" = ~ age, "formE2" = ~ age, "formE12" = ~ 1,
             "formA1" = ~ age, "formA2" = ~ 1, "formA12" = ~ 1)
Z_biv4 <- mt_twin(N_DZ = 327, N_MZ = 534, n_resp = 2, model = "AE",
                  formula = biv4, data = anthro)

## -----------------------------------------------------------------------------
## Initial values
control_initial <- list()
control_initial$regression <- list("R1" = c(0.13, 0.10, -0.20, -0.02, 0.037),
                                   "R2" = c(0.23, 0.01, -0.27, -0.11, 0.11))
control_initial$power <- list(c(0), c(0))
control_initial$tau <- c(0.15, 0, 0.12, rep(0,15))

fit_0 <- mglm4twin(linear_pred = c(form_Wt, form_Ht), matrix_pred = Z_biv0,
                          control_initial = control_initial,
                          control_algorithm = list(tuning = 0.5),
                          power_fixed = c(TRUE, TRUE),
                          data = anthro)

control_initial$tau <- c(0.15, 0, 0.12, rep(0,6))
fit_4 <- mglm4twin(linear_pred = c(form_Wt, form_Ht), matrix_pred = Z_biv4,
                   control_initial = control_initial,
                   control_algorithm = list(tuning = 0.5),
                   power_fixed = c(TRUE, TRUE),
                   data = anthro)

## ----warning=FALSE, message=FALSE---------------------------------------------
est0 <- aux_summary(fit_0, formula = biv0, type = "otimist", data = anthro)[,c(1,2,3,4)]
est4 <- aux_summary(fit_4, formula = biv4, type = "otimist", data = anthro)[,c(1,2,3,4)]
Table5 <- data.frame("Estimates" = est0$Estimates, 
                     "Std.error" = est0$Std..Error, 
                     "Z.value" = est0$Z.value,
                     "Estimates" = c(est4$Estimates[1:5], NA, est4$Estimates[6:8], 
                                     NA,est4$Estimates[9], rep(NA, 7)),
                     "Std.error" = c(est4$Std..Error[1:5], NA, est4$Std..Error[6:8], NA,  
                                     est4$Std..Error[9], rep(NA, 7)), 
                     "Z.value" = c(est4$Z.value[1:5], NA, est4$Z.value[6:8], NA, 
                                   est4$Z.value[9], rep(NA, 7)))
rownames(Table5) <- c("E1", "E1.age", "E2", "E2.age", "E12", "E12.age","A1","A1.age",
                      "A2", "A2.age", "A12", "A12.age", "C1", "C1.age", "C2", "C2.age",
                      "C12", "C12.age")
Table5 <- round(Table5, 2)

## -----------------------------------------------------------------------------
Table5

## -----------------------------------------------------------------------------
## Computing measures of interest
Age = seq(-1.90, 1.70, l = 100)
point <- fit_4$Covariance

# Heritability
h_wt <- (point[6] + point[7]*Age)/(point[1] + point[2]*Age + point[6] + point[7]*Age)
h_ht <- (point[8])/(point[3] + point[4]*Age + point[8])
h_ht_wt <- point[9]/(point[5] + point[9])

# Genetic and phenotipic correlation
r_G <- point[9]/(sqrt(point[6] + point[7]*Age)*sqrt(point[8]))
std_1 <- sqrt(point[1] + point[2]*Age + point[6] + point[7]*Age)
std_2 <- sqrt(point[3] + point[4]*Age + point[8])
r_P <- (point[9] + point[5])/(std_1*std_2)

## -----------------------------------------------------------------------------
require(mvtnorm)
Age = seq(-1.90, 1.70, l = 50)

# Point estimates
point1 <- fit_4$Covariance

# Covariance matrix (otimist - Gaussian case -> maximum likelihood estimator)
COV <- -2*fit_4$joint_inv_sensitivity[11:19, 11:19]

## Simulating from the assymptotic ditribution
point <- rmvnorm(n = 10000, mean = point1, sigma = as.matrix(COV))
res_h_wt <- matrix(NA, nrow = 50, ncol = 10000)
res_h_ht <- matrix(NA, nrow = 50, ncol = 10000)
res_ht_wt <- matrix(NA, nrow = 50, ncol = 10000)
res_r_G <- matrix(NA, nrow = 50, ncol = 10000)
res_r_P <- matrix(NA, nrow = 50, ncol = 10000)
res_r_E <- matrix(NA, nrow = 50, ncol = 10000)

for(i in 1:10000) {
  h_wt <- (point[i,6] + point[i,7]*Age)/(point[i,1] + point[i,2]*Age + point[i,6] + point[i,7]*Age)
  res_h_wt[,i] <- h_wt
  h_ht <- (point[i,8])/(point[i,3] + point[i,4]*Age + point[i,8])
  res_h_ht[,i] <- h_wt
  h_ht_wt <- point[i,9]/(point[i,5] + point[i,9])
  res_ht_wt[,i] <- h_ht_wt
  r_G <- point[i,9]/(sqrt(point[i,6] + point[i,7]*Age)*sqrt(point[i,8]))
  res_r_G[,i] <- r_G
  std_1 <- sqrt(point[i,1] + point[i,2]*Age + point[i,6] + point[i,7]*Age)
  std_2 <- sqrt(point[i,3] + point[i,4]*Age + point[i,8])
  r_P <- (point[9] + point[5])/(std_1*std_2)
  res_r_P[,i] <- r_P
  r_E <- point[i,5]/(sqrt(point[i,1] + point[i,2]*Age)*sqrt(point[i,3] + point[i,4]*Age))
  res_r_E[,i] <- r_E
}

## -----------------------------------------------------------------------------
Estimates <- c(rowMeans(res_h_wt), rowMeans(res_h_ht), 
               rowMeans(res_ht_wt),
               rowMeans(res_r_G), rowMeans(res_r_E),
               rowMeans(res_r_P))
Ic_Min <- c(apply(res_h_wt, 1, quantile, 0.025),
            apply(res_h_ht, 1, quantile, 0.025),
            apply(res_ht_wt, 1, quantile, 0.025),
            apply(res_r_G, 1, quantile, 0.025),
            apply(res_r_E, 1, quantile, 0.025),
            apply(res_r_P, 1, quantile, 0.025))
Ic_Max <- c(apply(res_h_wt, 1, quantile, 0.975),
            apply(res_h_ht, 1, quantile, 0.975),
            apply(res_ht_wt, 1, quantile, 0.975),
            apply(res_r_G, 1, quantile, 0.975),
            apply(res_r_E, 1, quantile, 0.975),
            apply(res_r_P, 1, quantile, 0.975))
data_graph <- data.frame("Estimates" = Estimates, "Ic_Min" = Ic_Min,
                         "Ic_Max" = Ic_Max, "Age" = rep(Age, 6), 
                         "Parameter" = rep(c("h1","h2","h12","G","E","P"), each = 50))

## -----------------------------------------------------------------------------
require(lattice)
require(latticeExtra)
source("https://raw.githubusercontent.com/walmes/wzRfun/master/R/panel.cbH.R")
data_graph$Parameter <- factor(data_graph$Parameter, 
                               levels = c("h1","h2","h12","G","E","P"))
levels(data_graph$Parameter) <- c("Weight heritability", 
                                  "Height heritability", 
                                  "Biv. heritability",
                                  "Genetic correlation",
                                  "Environmental correlation",
                                  "Phenotypic correlation")

ylim = list("1" = c(0.75,1), "1" = c(0.75, 1),
            "3" = c(0.75,1), "2" = c(0.1, 1),
            "3" = c(0.1,1), "3" = c(0.1, 1))

xy1 <- xyplot(Estimates ~ Age | Parameter, data = data_graph, 
              ylim = ylim, 
              type = "l", ly = data_graph$Ic_Min, 
              xlab = c("Standardized age"),
              uy = data_graph$Ic_Max, scales = "free",
              cty = "bands", alpha = 0.25, prepanel = prepanel.cbH, 
              panel = panel.cbH)

## ----echo = FALSE, out.width='99%', message=FALSE, fig.width=10, fig.height=6, fig.align='center', fig.cap = 'Genetic, environment and phenotypic correlations (first line). Body weight heritability, height heritability and bivariate heritability (second line) - Bivariate continuous Twin data.'----
xy1

