## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----eval=FALSE---------------------------------------------------------------
#  library(devtools)
#  install_git("bonatwagner/mglm4twin")

## ----eval=TRUE, error=FALSE, message=FALSE, warning=FALSE---------------------
library(mglm4twin)
packageVersion("mglm4twin")

## -----------------------------------------------------------------------------
## Loading extra packages
require(Matrix)
require(SimCorMultRes)
require(mglm4twin)

## Setting model parameters
E <- c(0.75, 0.7, 0.65, -0.3, 0.25, -0.4)
A <- c(0.25,0.3, 0.35, -0.15, 0.20, -0.2)
C <- rep(0, 6)
tau = c(E, A, C)

## Twin structure
DZ = mt_twin(N_DZ = 1, N_MZ = 0, n_resp = 3, model = "ACE")
MZ = mt_twin(N_DZ = 0, N_MZ = 1, n_resp = 3, model = "ACE")
Omega_DZ <- as.matrix(mt_matrix_linear_predictor(tau = tau, Z = DZ))
Omega_MZ <- as.matrix(mt_matrix_linear_predictor(tau = tau, Z = MZ))

## Regression structures
sex <- c(rep("Male", 220/2), rep("Female", 334/2), 
         rep("Male", 116/2), rep("Female", 158/2))
zyg <- c(rep("DZ", 554/2), rep("MZ", 274/2))
set.seed(123)
Age <- rbeta(414, shape1 = 0.3*2, shape2 = 0.7*2)*20 + 70
age_std <- (Age - mean(Age))/var(Age)
X <- model.matrix(~ sex + age_std)
beta1 <- c(1.6956, 0.0584, -0.2576)
mu1 <- exp(X%*%beta1)/(1+exp(X%*%beta1))
beta2 <- c(-1.7930, 0.0875, 0.2382)
mu2 <- exp(X%*%beta2)/(1+exp(X%*%beta2))
beta3 <- c(-0.5363, -0.05138, -0.1528)
mu3 <- exp(X%*%beta3)/(1+exp(X%*%beta3))

## Marginal distributions
dist <- c("qbeta","qbeta","qbeta")
invcdfnames <- rep(dist, each = 2)

## Simulating data set
Y1_MZ <- list()
Y2_MZ <- list()
Y3_MZ <- list()

Y1_DZ <- list()
Y2_DZ <- list()
Y3_DZ <- list()
set.seed(181185)
phi = 5
for(i in 1:137) {
  paramslists <- list(
    m1 = list(shape1 = mu1[i]*phi, shape2 = (1-mu1[i])*phi),
    m1 = list(shape1 = mu1[i]*phi, shape2 = (1-mu1[i])*phi),
    m2 = list(shape1 = mu2[i]*phi, shape2 = (1-mu2[i])*phi),
    m2 = list(shape1 = mu2[i]*phi, shape2 = (1-mu2[i])*phi),
    m3 = list(shape1 = mu3[i]*phi, shape2 = (1-mu3[i])*phi),
    m3 = list(shape1 = mu3[i]*phi, shape2 = (1-mu3[i])*phi))
  Y_MZ <- rnorta(R = 1, cor.matrix = Omega_MZ,
                 distr = invcdfnames, qparameters = paramslists)
  Y1_MZ[[i]] <- Y_MZ[1:2]
  Y2_MZ[[i]] <- Y_MZ[3:4]
  Y3_MZ[[i]] <- Y_MZ[5:6]
}
set.seed(181185)
for(i in 138:414) {
  paramslists <- list(
    m1 = list(shape1 = mu1[i]*phi, shape2 = (1-mu1[i])*phi),
    m1 = list(shape1 = mu1[i]*phi, shape2 = (1-mu1[i])*phi),
    m2 = list(shape1 = mu2[i]*phi, shape2 = (1-mu2[i])*phi),
    m2 = list(shape1 = mu2[i]*phi, shape2 = (1-mu2[i])*phi),
    m3 = list(shape1 = mu3[i]*phi, shape2 = (1-mu3[i])*phi),
    m3 = list(shape1 = mu3[i]*phi, shape2 = (1-mu3[i])*phi))
  Y_DZ <- rnorta(R = 1, cor.matrix = Omega_DZ,
                 distr = invcdfnames, qparameters = paramslists)
  Y1_DZ[[i]] <- Y_DZ[1:2]
  Y2_DZ[[i]] <- Y_DZ[3:4]
  Y3_DZ[[i]] <- Y_DZ[5:6]
}

Y1 <- c(do.call(c, Y1_DZ), do.call(c, Y1_MZ))
Y2 <- c(do.call(c, Y2_DZ), do.call(c, Y2_MZ))
Y3 <- c(do.call(c, Y3_DZ), do.call(c, Y3_MZ))

data <- data.frame("Y1" = Y1, "Y2" = Y2, "Y3" = Y3, "twin_id" = rep(1:2, 414),
                   "zyg" = rep(zyg, each = 2), "sex" = rep(sex, each = 2), 
                   "age_std" = rep(age_std, each = 2))

## -----------------------------------------------------------------------------
form_Y1 <- c(Y1 ~ sex + age_std)
form_Y2 <- c(Y2 ~ sex + age_std)
form_Y3 <- c(Y3 ~ sex + age_std)

## -----------------------------------------------------------------------------
ACE = mt_twin(N_DZ = 554/2, N_MZ = 274/2, n_resp = 3, model = "ACE")
AE = mt_twin(N_DZ = 554/2, N_MZ = 274/2, n_resp = 3, model = "AE")
CE = mt_twin(N_DZ = 554/2, N_MZ = 274/2, n_resp = 3, model = "CE")
E = mt_twin(N_DZ = 554/2, N_MZ = 274/2, n_resp = 3, model = "E")

## -----------------------------------------------------------------------------
link = rep("logit", 3)
variance = rep("binomialP", 3)

## ----warning = FALSE----------------------------------------------------------
fit_ACE <- mglm4twin(linear_pred = c(form_Y1, form_Y2, form_Y3), 
                     matrix_pred = ACE, link = link,
                     variance = variance, 
                     data = data)
fit_AE <- mglm4twin(linear_pred = c(form_Y1, form_Y2, form_Y3), 
                    matrix_pred = AE, link = link,
                    variance = variance, 
                    data = data)
fit_CE <- mglm4twin(linear_pred = c(form_Y1, form_Y2, form_Y3), 
                    matrix_pred = CE, link = link,
                    variance = variance, 
                    data = data)
fit_E <- mglm4twin(linear_pred = c(form_Y1, form_Y2, form_Y3), 
                    matrix_pred = E, link = link,
                    variance = variance, 
                    data = data)

## -----------------------------------------------------------------------------
rbind(gof(fit_E), gof(fit_AE), gof(fit_CE), gof(fit_ACE))[,c(1,2,3,5)]

## -----------------------------------------------------------------------------
summary(fit_AE, model = "AE", biometric = TRUE)

