## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----eval=FALSE---------------------------------------------------------------
#  library(devtools)
#  install_git("bonatwagner/mglm4twin")

## ----eval=TRUE, error=FALSE, message=FALSE, warning=FALSE---------------------
library(mglm4twin)
packageVersion("mglm4twin")

## -----------------------------------------------------------------------------
data(bmi)
head(bmi)

## -----------------------------------------------------------------------------
linear_pred <- bmi ~ Group*Twin_pair

## -----------------------------------------------------------------------------
ACE = mt_twin(N_DZ = 5576/2, N_MZ = 2966/2, n_resp = 1, model = "ACE")
AE = mt_twin(N_DZ = 5576/2, N_MZ = 2966/2, n_resp = 1, model = "AE")

## -----------------------------------------------------------------------------
bmi$age <- (bmi$age - mean(bmi$age))/sd(bmi$age)
list_form <- list(formE = ~ age + gender, formA = ~ age + gender,
                  formC = ~ age + gender)

## -----------------------------------------------------------------------------
ACE_reg = mt_twin(N_DZ = 5576/2, N_MZ = 2966/2, n_resp = 1,
                  model = "ACE", formula = list_form, data = bmi)

## -----------------------------------------------------------------------------
list_form2 <- list(formE = ~ age + gender, formA = ~ age + gender)
AE_reg = mt_twin(N_DZ = 5576/2, N_MZ = 2966/2, n_resp = 1,
                 model = "AE", formula = list_form2, data = bmi)

## -----------------------------------------------------------------------------
link = "identity"
variance = "constant"

## -----------------------------------------------------------------------------
## Standard ACE model
fit_ACE <- mglm4twin(linear_pred = c(linear_pred), 
                     matrix_pred = ACE, data = bmi)

## Standard AE model
fit_AE <- mglm4twin(linear_pred = c(linear_pred), 
                    matrix_pred = AE, data = bmi)

## ACE regression on the dispersion
fit_ACE_reg <- mglm4twin(linear_pred = c(linear_pred), 
                         matrix_pred = ACE_reg, data = bmi)

## AE regression on the dispersion
fit_AE_reg <- mglm4twin(linear_pred = c(linear_pred), 
                        matrix_pred = AE_reg, data = bmi)

## -----------------------------------------------------------------------------
mt_anova_mglm(fit_ACE_reg, formula = list_form, data = bmi)

## -----------------------------------------------------------------------------
mt_anova_mglm(fit_AE_reg, formula = list_form2, data = bmi)

## ----eval = FALSE-------------------------------------------------------------
#  aux_summary(fit_AE_reg, formula = list_form2,
#              type = "robust2", id = bmi$Twin, data = bmi)

## -----------------------------------------------------------------------------
list_form3 <- list(formE = ~ age + gender, formA = ~ gender)
AE_reg1 = mt_twin(N_DZ = 5576/2, N_MZ = 2966/2, n_resp = 1,
                  model = "AE", formula = list_form3, data = bmi)
fit_AE_reg1 <- mglm4twin(linear_pred = c(linear_pred), matrix_pred = AE_reg1,
                         data = bmi)

## ----echo = FALSE, results = "hide"-------------------------------------------
Parameters <- c("E0","E_age","E_gender", "A0", "A_age", "A_gender",
                "C0", "C_age", "C_gender", "plogLik(df)", "pAIC")

## ACE regression model
tt = aux_summary(fit_ACE_reg, formula = list_form, type = "otimist", 
                 id = bmi$Twin, data = bmi)
std_ACE_reg <- tt$Std..Error
ACE_reg <- fit_ACE_reg$Covariance

## AE regression model
AE_reg <- c(fit_AE_reg$Covariance, rep(NA, 3))
tt = aux_summary(fit_AE_reg, formula = list_form2, type = "otimist", data = bmi)
std_AE_reg <- tt$Std..Error
std_AE_reg <- c(std_AE_reg, rep(NA, 3))

## AE simplified 
AE_reg1 <- c(fit_AE_reg1$Covariance[1:3], fit_AE_reg1$Covariance[4], NA,
             fit_AE_reg1$Covariance[5], rep(NA, 3))
tt = aux_summary(fit_AE_reg1, formula = list_form3, type = "otimist", data = bmi)
std_AE_reg1 <- tt$Std..Error
std_AE_reg1 <- c(std_AE_reg1[1:3], std_AE_reg1[4], NA, std_AE_reg1[5], rep(NA, 3))

## Table 2
Table2 <- data.frame(round(cbind(ACE_reg, std_ACE_reg,AE_reg, std_AE_reg, AE_reg1, std_AE_reg1),2))
LL <- c(plogLik(fit_ACE_reg, verbose = FALSE)$plogLik, 
        plogLik(fit_ACE_reg, verbose = FALSE)$df,
        plogLik(fit_AE_reg, verbose = FALSE)$plogLik, 
        plogLik(fit_AE_reg, verbose = FALSE)$df,
        plogLik(fit_AE_reg1, verbose = FALSE)$plogLik, 
        plogLik(fit_AE_reg1, verbose = FALSE)$df)

pAIC <- c(pAIC(fit_ACE_reg, verbose = FALSE)$pAIC, NA, 
          pAIC(fit_AE_reg, verbose = FALSE)$pAIC, NA, 
          pAIC(fit_AE_reg1, verbose = FALSE)$pAIC, NA)
Table2 <- rbind(Table2, LL, pAIC)
Table2 <- cbind(Parameters, Table2)

## -----------------------------------------------------------------------------
Table2

## ----echo = FALSE, out.width='99%', message=FALSE, fig.width=5, fig.height=5, fig.align='center', fig.cap = 'Environment, additive genetic and heritability index as a function of standardized age.'----
phiE = fit_AE_reg1$Covariance[1:3]
phiA = fit_AE_reg1$Covariance[4:5]
data_new <- data.frame("age" = rep(seq(from = -1.539967, to = 2.098272, length = 30), 2),
                       "gender" = rep(c("female", "male"), each = 30))
X_E <- model.matrix(~ age + gender, data = data_new)
X_A <- model.matrix(~ gender, data = data_new)
Point <- fit_AE_reg1$Covariance
VCOV <- fit_AE_reg1$vcov[5:9, 5:9]

## Simulation (computin standard  errors and confidence interval)
require(mvtnorm)
require(lattice)
require(latticeExtra)
source("https://raw.githubusercontent.com/walmes/wzRfun/master/R/panel.cbH.R")
tt = rmvnorm(n = 5000, mean = Point, sigma = as.matrix(VCOV))

# Computing measures of interest
tauE <- matrix(NA, nrow = dim(X_E)[1], ncol = 5000)
tauA <- matrix(NA, nrow = dim(X_E)[1], ncol = 5000)
h <- matrix(NA, nrow = dim(X_E)[1], ncol = 5000)

for(i in 1:5000) {
  tauE[,i] <- X_E%*%tt[i,1:3]
  tauA[,i] <- X_A%*%tt[i,4:5]
  h[,i] <- tauA[,i]/(tauE[,i] + tauA[,i])
}

# Environment
data_new$tauE_est <- apply(tauE, 1, mean)
tauE_IC <- apply(tauE, 1, quantile, c(0.025, 0.975))
data_new$tauE_IcMin <- tauE_IC[1,]
data_new$tauE_IcMax <- tauE_IC[2,]

# Genetic
data_new$tauA_est <- apply(tauA, 1, mean)
tauA_IC <- apply(tauA, 1, quantile, c(0.025, 0.975))
data_new$tauA_IcMin <- tauA_IC[1,]
data_new$tauA_IcMax <- tauA_IC[2,]

# Heritability
conf_h = data.frame(t(apply(h, 1, quantile, c(0.025, 0.975))))
data_new$IcMin_h <- conf_h[,1]
data_new$IcMax_h <- conf_h[,2]
data_new$h <- rowMeans(h)
levels(data_new$gender) <- c("Female", "Male")

data_graph <- with(data_new,
                   data.frame("Estimates" = c(h, tauE_est, tauA_est),
                              "Ic_Min" = c(IcMin_h, tauE_IcMin, tauA_IcMin),
                              "Ic_Max" = c(IcMax_h, tauE_IcMax, tauA_IcMax),
                              "Age" = rep(age, 3), "gender" = rep(gender, 3),
                              "Parameter" = rep(c("h","tauE","tauA"), each = 60) ))
levels(data_graph$Parameter) <- c("Heritability", "Genetic", "Environment")
ylim = list("1" = c(0.5,0.9), "1" = c(0.5, 0.9),
            "2" = c(6,12),"2" = c(6, 12),
            "3" = c(1, 7), "3" = c(1, 7))
xy1 <- xyplot(Estimates ~ Age | gender + Parameter, data = data_graph,
              ylim = ylim, type = "l", ly = data_graph$Ic_Min,
              xlab = c("Standardized age"),
              uy = data_graph$Ic_Max, scales = "free",
              #par.settings = ps,
              cty = "bands", alpha = 0.25, 
              prepanel = prepanel.cbH,
              panel = panel.cbH)
useOuterStrips(combineLimits(xy1,adjust.labels = TRUE))

