/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

#include "aRTconn.h"
#include "aRTdb.h"
#include "SEXPbasic.h"

#include <TeUpdateDBVersion.h>

#include <string>
#include <stdlib.h>

#ifdef HAVE_MYSQL
	#include <TeMySQL.h>
#endif

#ifdef HAVE_POSTGRE 
	#include <TePGInterface.h>
	#include <TePGUtils.h>
	#include <TePostgreSQL.h>
#endif

extern "C"
{

SEXP aRTconn::AddUser(SEXP data)
{
	string username = GET_STRING_ELEMENT(data,"userName");

	PrintSilent("Adding user \'%s\' ... ", username.c_str());
	TeDatabase* database = NewTeDatabase();

	if( !database -> connect(Host, User, Password, "", Port))
		error("Could not connect\n");

	stringstream stream;
	stream << "GRANT ALL PRIVILEGES ON *.* TO " << username << "@localhost;";
	string sql = StreamToChar(stream);
	if( !database->execute(sql) )
	{
		string err = database->errorMessage();
		delete database;
		error( err.c_str() );
	}
	
	delete database;
    PrintSilentYes;
	return RNULL;
}

SEXP aRTconn::DeleteDb(SEXP data)
{
	string databasename = GET_STRING_ELEMENT(data, "dbname");

	// "checking ...": R code already does.
	TeDatabase* database = NewTeDatabase();

	if( !database -> connect(Host, User, Password, "", Port))
	{
		PrintSilentNo;
		error("Could not connect\n");
	}
	
	PrintSilent("Deleting database \'%s\' ... ", databasename.c_str());
	
	stringstream stream;
	stream << "DROP DATABASE " << databasename << ";";
	string sql = StreamToChar(stream);

	if( !database->execute(sql) )
	{
		delete database;
		PrintSilentNo;
		error( database->errorMessage().c_str() );
	}
	
	delete database;
    PrintSilentYes;
	return RNULL;
}

SEXP aRTconn::List()
{
	TeDatabase* database = NewTeDatabase();	
	vector<string> db_names;
	SEXP vectorDbs;

	if (!database -> showDatabases (Host, User, Password, db_names, Port))
	{
		delete database;
		return NULL;
	}

	vectorDbs = allocVector(STRSXP, db_names.size()); // Vector containing the database names

	for (unsigned i = 0; i != db_names.size(); i++)
	  SET_STRING_ELT(vectorDbs, i, mkChar(db_names[i].c_str()));

	delete database;
	return vectorDbs;
}

aRTconn::aRTconn(string user, string pass, uint port, string host, string dbms)
{
	PrintSilent("Trying to connect ... ");
	User = user;
	Password = pass;
	Port = port;
	Host = host;
	
	if     (dbms == "mysql")   Type = aRTmySQL;
	else if(dbms == "postgre") Type = aRTpostgres;
	else error("Invalid database type: %s\n", dbms.c_str());

	TeDatabase* database = NewTeDatabase();
	vector<string> db_names;

	// the object is valid if we can list the databases
	bool Valid = database -> showDatabases(Host, User, Password, db_names, Port);
	
	if (!Valid)
	{
		PrintSilentNo;
		string error_msg = database -> errorMessage();
		delete database;
		error(error_msg.c_str());
	}
	delete database;
    PrintSilentYes;
}

SEXP aRTconn::Print()
{
	stringstream s;
	string ret;
	string dbms;

    if(Type == aRTmySQL)    dbms = "MySQL";
    if(Type == aRTpostgres) dbms = "PostgreSQL";
	
	s << artOBJECT("aRTconn")
	  << "\n\nDBMS: \""     << dbms
	  << "\"\nUser: \""     << User
	  << "\"\nPassword: \"" << Password
	  << "\"\nPort: "       << Port
	  << "\nHost: \""       << Host << "\"" << endl;
	
	Rprintf( StreamToChar(s) );
	return RNULL;
}

// aRTSummary
enum aRTconnSenum
{
	aRTconnSClass = 0,
	aRTconnSUser,
	aRTconnSPassword,
	aRTconnSPort,
	aRTconnSHost,
	aRTconnSDatabases,
	aRTconnSSum
};

SEXP aRTconn::Summary()
{
	SEXP info, element, names;

	info = allocVector(VECSXP, aRTconnSSum); // List containing the connection attributes
	names = allocVector(STRSXP, aRTconnSSum); // Names Vector

    STRING_LIST_ITEM(info, names, aRTconnSClass,     "class",    Class());
    STRING_LIST_ITEM(info, names, aRTconnSUser,      "user",     User);
    STRING_LIST_ITEM(info, names, aRTconnSPassword,  "password", Password);
    STRING_LIST_ITEM(info, names, aRTconnSHost,      "host",     Host);

    INTEGER_LIST_ITEM(info, names, aRTconnSPort,     "port",     Port);

    SET_STRING_ELT(names, aRTconnSDatabases, mkChar("databases"));
    element = List();
    SET_VECTOR_ELT(info, aRTconnSDatabases, element);

	setAttrib(info, R_NamesSymbol, names); // Set the names of the list
	
	return info;		 
}

// Checks the database type and return a pointer to it.
TeDatabase* aRTconn::NewTeDatabase()
{
	switch(Type)
	{
#ifdef HAVE_MYSQL 
		case aRTmySQL:
			return new TeMySQL();
#endif
#ifdef HAVE_POSTGRE
		case aRTpostgres:
			return new TePostgreSQL();
#endif
		default:
				return NULL;
	}
	return NULL;
}

// Try to open an existent database else create a new database if (create)
aRTcomponent* aRTconn::Open(SEXP data)
{
	bool silent = Silent; // **

    string dbname = GET_STRING_ELEMENT(data, "dbname");
	bool create   = GET_BOOL_ELEMENT(data,   "create");
	bool update   = GET_BOOL_ELEMENT(data,   "update");

	PrintSilent("Connecting with database \'%s\' ... ", dbname.c_str());
	
	Silent = false; // **
	TeDatabase* database = NewTeDatabase();

	// ** things doesn't work if this line executes with Silent == true
	// ** strange... I already have tried to change the name of this variable.
    if (database -> connect(Host, User, Password, dbname, Port))
	{
		Silent = silent; // **
		PrintSilentYes;
        if(create) warning("Trying to create an existent database\n");
		
		string version, error_msg;
		if( needUpdateDB(database, version) )
		{
			if(update)
			{
				PrintSilent("Updating database \'%s\' to new terraLib version ... ", dbname.c_str());
				if( updateDBVersion(database, version, error_msg) )
					PrintSilentYes;
				else
					error("Could not update the database version: %d\n", error_msg.c_str());
			}
			else error("Database version needs to be updated. Set update=TRUE\n");
		}
	}
	else
	{
		Silent = silent; // **
		PrintSilentNo;
		if(!create)
			error("Database \'%s\'does not exist. If you want to create set create=TRUE\n", dbname.c_str());

		PrintSilent("Creating database \'%s\' ... ", dbname.c_str());
        if (!database -> newDatabase(dbname, User, Password, Host, Port))
		{
	        string error_ = database -> errorMessage();
            delete database;
			error(error_.c_str());
	    }
		PrintSilentYes;
		PrintSilent("Creating conceptual model of database \'%s\' ... ", dbname.c_str());
		if (!database->createConceptualModel(true, false))
		{
			string error_ = database -> errorMessage();
			delete database;
			error(error_.c_str());
		}
		PrintSilentYes;
	}
	
	aRTdb* root = new aRTdb(database);
	return root;
}

}

