#  aRT : API R - TerraLib                                                
#  Copyright (C) 2003-2005  LEG                                          
#                                                                        
#  This program is free software; you can redistribute it and/or modify  
#  it under the terms of the GNU General Public License as published by  
#  the Free Software Foundation; either version 2 of the License, or     
#  (at your option) any later version.                                   
#                                                                        
#  This program is distributed in the hope that it will be useful,       
#  but WITHOUT ANY WARRANTY; without even the implied warranty of        
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         
#  GNU General Public License for more details.                          
#                                                                        
#  You should have received a copy of the GNU Lesser General Public      
#  License along with this library.

setMethod("initialize", "aRTlayer", function(.Object,
                                             aRTdb,
											 layername,
											 create = FALSE,
											 proj = NA)
{
  if (!is.object(aRTdb) || class(aRTdb) != "aRTdb")
    stop("Argument \"aRTdb\" must be an object of aRTdb\n")
  if (is.null(layername)) stop("layername can not be NULL")

  if(is.na(proj)) proj = ""

  out <- .aRTcall(aRTdb,
                  "cppOpen",
                  layername=layername,
			      create=as(create, "integer"),
			      projection=proj)

  if (is.null(out))
    stop("Could not create the aRTlayer object\n")
  .Object@pointer <- out
  reg.finalizer(.Object@pointer, .aRTremove)
  .Object
})

setMethod("showTables", "aRTlayer", function(object) { .aRTcall(object, "cppLShowTables") })

# import a static table
setMethod("importTable", "aRTlayer", function(object, table, id, data)
{
	if(is(data, "Spatial")) data = data@data@att

	t = new("aRTtable", object, table=table, id=id, c=TRUE, gen=FALSE)

	colnames = names(data)
    classes = substr(sapply(data, class),1,1)
    for(i in 1:length(colnames))
    {
		if(colnames[i] != id)
		{
			createColumn(t,
			             colnames[i],
						 type = classes[i],
						 length=max(nchar(as.matrix(data[i]))))
		}
    }
	addRows(t, data.frame(data))
	t
})

setMethod("addShape", "aRTlayer", function(object, file, table, id, length=10)
{
	.aRTcall(object,
	         "cppAddShape", 
	         filename = file,
 	         tablename = table,
	 	     id = id,
			 length = as.integer(length))
	return(invisible())
})

setMethod("addPoints", "aRTlayer", function(object, points) {
    if(is(points,"SpatialPointsDataFrame")) points = as.aRTgeometry(points)

    if(class(points) != "aRTgeometry")
        stop("'points' is an invalid object")
				
	.Call("cppAddPoints", object@pointer, points@pointer, PACKAGE="aRT")
	return(invisible())
})

setMethod("addPolygons", "aRTlayer", function(object, polygons) {
    if(is(polygons,"SpatialPolygons")) polygons = as.aRTgeometry(polygons) 

	if(class(polygons) != "aRTgeometry")
		stop("'polygons' is an invalid object")
	
	.Call("cppAddPolygons", object@pointer, polygons@pointer, PACKAGE="aRT")
	return(invisible())
})

setMethod("addLines", "aRTlayer", function(object, lines) {
	if(is(lines, "SpatialLines")) lines = as.aRTgeometry(lines)
	
	if(class(lines) != "aRTgeometry")
	        stop("'lines' is an invalid object")
			
	.Call("cppAddLines", object@pointer, lines@pointer, PACKAGE="aRT")

	return(invisible())
})

setMethod("addRaster", "aRTlayer", function(object, value) {
  if(summary(object)$raster == "yes") stop("the layer already has raster data")
  if(!any(names(value)=="z"))         stop ("'value' must have a z attribute")
  if(!any(names(value)=="x"))         stop ("'value' must have a x attribute")
  if(!any(names(value)=="y"))         stop ("'value' must have a y attribute")

  .aRTcall(object, "cppAddRaster",
           x=range(value$x),
           y=range(value$y),
           z=value$z,
           dim = dim(value$z))

  return(invisible())
})

setMethod("getProj", "aRTlayer", function(object) {
  .aRTcall(object, "cppGetProj")
})

setMethod("getBox", "aRTlayer", function(object) {
  .aRTcall(object, "cppGetBox")
})

setMethod("getNN", "aRTlayer", function(object, layer=object, id, quantity=1) {
	.Call("cppGetNearestNeighbors",
		  object@pointer,
		  list(id=id,
		       quantity=quantity),
		  layer@pointer,
		  PACKAGE="aRT")
})

setMethod("getDistance", "aRTlayer", function(object, id, layer=object) {
#	if(!is.null(layer)) stop("Distances between tow layers not supported yet")
	.Call("cppGetDistance",
	      object@pointer,
		  id,
		  layer@pointer,
		  PACKAGE="aRT")
})

setMethod("getRelation", "aRTlayer", function(object,
                                              relation = c("contains",
											               "overlaps",
    													   "crosses",
														   "disjoint",
														   "touches",
														   "within",
														   "intersects",
														   "equals",
														   "covers",
														   "coveredby"),
											  layer=object,
											  id=NULL) {
	relation = match.arg(relation)
	result = .Call("cppGetRelation",
                   object@pointer,
                   list(relation=relation,
                        id=id),
		           layer@pointer,
                   PACKAGE="aRT")
	unique(result)
})

setMethod("getSetOperation", "aRTlayer",
function(object,
         operation = c("union",
		               "intersection",
					   "difference",
					   "xor"),
         id,
		 as.sp = TRUE)
{
  operation = match.arg(operation)
  result <- .aRTcall(object, "cppGetSetOperation", operation=operation, id=id)

  data = new("aRTgeometry", .external = result)
  if(as.sp) data = getGeometry(data)
  data
})

setMethod("getOperation", "aRTlayer",
function(object,
         operation = c("centroid",
				       "hull",
					   "buffer"),
		 id = NULL,
		 distance = NULL,
		 as.sp = TRUE)
{

  operation = match.arg(operation)
  result <- .aRTcall(object, "cppGetOperation", operation=operation, dist=distance, id=id)

  data = new("aRTgeometry", .external = result)
  if(as.sp) data = getGeometry(data)
  data
})


setMethod("getMetric", "aRTlayer", function(object,
                                            metric = c("area",
											           "length",
													   "distance",
													   "withinDist"),
											id = NULL,
											distance = NULL){
#											origin = NULL) {
  metric = match.arg(metric)
  .aRTcall(object, "cppGetMetric", metric=metric, dist=distance, id=id)
})

setMethod("getPoints", "aRTlayer", function(object, table=NULL, as.sp) {
	if(!is.null(table)) as.sp = TRUE

	su = summary(object)
    if(su$points == 0) return(NULL)

	q = openQuerier(object, geom = "points")
	data = getData(q, as.sp=as.sp)

	if( !is.null(table) )
	{
		if( length(summary(table)$keys) > 1)
			stop("Table must have only one unique id in this function")

		d = getData(table)

		data.merged=merge(data.frame(data), d, by.x="ID", by.y=summary(table)$keys)
		coordinates(data.merged) = c("coords.x1","coords.x2")
		data = data.merged
	}

	return(data)
})

setMethod("getLines", "aRTlayer", function(object, as.sp) {
	su = summary(object)
	if(su$lines == 0) return(NULL)

	q = openQuerier(object, geom = "lines")
	data = getData(q, as.sp=as.sp)

	return(data)
})

setMethod("getPolygons", "aRTlayer", function(object, as.sp) {
	su = summary(object)
	if(su$polygons == 0) return(NULL)

	q = openQuerier(object, geom = "polygons")
	data = getData(q, as.sp=as.sp)

	return(data)
})

setMethod("getRaster", "aRTlayer", function(object) {
  .aRTcall(object, "cppGetRaster")
})

setMethod("openQuerier", "aRTlayer", function(object, geom = c("", "points", "lines", "polygons"))
{
  geom = match.arg(geom)
  err = "Layer has more than one geometry, choose one value for geom"
  if(geom == "") # if there is more than one geometry and the user
                 # does not specify the geom we stop with an error
  {
    sum = summary(object)

    if(sum$polygons > 0) geom = "polygons"
    if(sum$lines > 0   ) { if(geom == "") geom = "lines"  else stop(err) }
    if(sum$points > 0  ) { if(geom == "") geom = "points" else stop(err) }
    if(geom == "") stop("Layer does not have any geometry")
  }

  new("aRTquerier", object, loadatt=FALSE, geom=geom)
})

setMethod("openTable", "aRTlayer", function(object, tablename=NULL, type="default") {
	if(type=="media")
		ret = new("aRTtable", object, type=type)
	else
	{	
		if(!any(showTables(object) == tablename))
    	    stop("Table does not exist\n")
		ret = new("aRTtable", object, tab=tablename, type=type)
	}
	return(ret)
})

setMethod("createTable", "aRTlayer", function(object,
                                              tablename=NULL,
                                              genids=NULL,
											  type="static",
											  id="id",
											  ...) {
	if(type=="media")
		ret = new("aRTtable", object, type=type, c=TRUE, ...)
	else
	{	
		if(any(showTables(object) == tablename))
        	stop("Table already exists\n")
		ret = new("aRTtable", object, tab=tablename, gen=genids, type=type, id=id, c=TRUE, ...)
	}
	return(ret)
})


