/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

/* In this file, we intent to convert some pointers to C++
 * objects and to call their functions. 
 * 
 * All objects in R contains only the pointer to TerraLib data.
 * This code only converts SEXPs to Te* and Te* to SEXPs, 
 * checks if the data is valid, and calls the functions.
 */

#include "aRTcomponent.h"
#include "aRT.h"
#include "aRTdb.h"
#include "aRTlayer.h"
#include "aRTtheme.h"
#include "aRTdlcommon.h"
#include "SEXPutils.h"
#include "SEXPbasic.h"

extern "C" {
#include <Rdefines.h>
#include <Rinternals.h>

SEXP getObjHandle(SEXP obj)
{
    SEXP sxpHandle = getAttrib( obj, mkString("pointer") );
    if( isNull(sxpHandle) ) error("Object is NULL");
    return sxpHandle;
}

void* getObj(SEXP sxpHandle)
{
    void* pointer = R_ExternalPtrAddr(sxpHandle);
    if(!pointer) error("External pointer is NULL");
    return pointer;
}

// converts a SEXP SexpValue_ to a Type_* and puts it into Pointer_
#define SEXP_TO_ARTP(SexpValue_, Type_, Pointer_) \
  Pointer_ = (Type_*) getObj(SexpValue_);

#define ARTP_TO_SEXP(Pointer_, SexpValue_)                                     \
  if (Pointer_ == NULL) SexpValue_ = R_NilValue;                               \
  else                                                                         \
  {                                                                            \
    SexpValue_ = R_MakeExternalPtr((void*) Pointer_, mkChar("aRT"),R_NilValue);\
  }

// check if a SEXP Pointer_ is a valid aRTcomponent*.  If not, returns Return_.
// This macro must be inside a function.
#define CHECK_VALID(SPointer_)                              \
{                                                           \
    aRTcomponent* art_component_;                           \
    SEXP_TO_ARTP(SPointer_, aRTcomponent, art_component_);  \
    if( !art_component_ -> IsValid() ) {                    \
        error("Invalid object");                            \
    }                                                       \
}

// declares a function called FuncionName, that receives a SEXP Pointer
// as argument. It converts the Pointer to a Type*, calls the function 
// Call and then prints its returned value.

#define PRINT_FUNCTION_0(FunctionName_, Type_, Call_) \
SEXP FunctionName_(SEXP SPointer)                     \
{                                                     \
    CHECK_VALID(SPointer)                             \
    Type_* obj;                                       \
    SEXP_TO_ARTP(SPointer, Type_, obj);               \
    string result = obj -> Call_;                     \
    Rprintf("%s", result.c_str());                    \
    return R_NilValue;                                \
}

#define VOID_FUNCTION_1(FunctionName_, arg_, Type_, Call_)  \
SEXP FunctionName_(SEXP Pointer, SEXP arg_)                 \
{                                                           \
    CHECK_VALID(Pointer);                                   \
    Type_* obj;                                             \
    SEXP_TO_ARTP(Pointer, Type_, obj);                      \
    obj->Call_;                                             \
    return R_NilValue;                                      \
}

#define SEXP_FUNCTION_0(FunctionName_, Type_, Call_) \
	SEXP FunctionName_(SEXP SPointer) {              \
    CHECK_VALID(SPointer);                           \
    Type_* obj;                                      \
    SEXP_TO_ARTP(SPointer, Type_, obj)               \
    return obj -> Call_;                             \
}

#define SEXP_FUNCTION_1(FunctionName_, arg_, Type_, Call_)  \
SEXP FunctionName_(SEXP Pointer, SEXP arg_)                 \
{                                                           \
    CHECK_VALID(Pointer);                                   \
    Type_* obj;                                             \
    SEXP_TO_ARTP(Pointer, Type_, obj);                      \
    return obj->Call_;                                      \
}

// a function that gets two pointers as args
#define SEXP_FUNCTION_2P(FunctionName_, Type_, Call_)                \
SEXP FunctionName_(SEXP SPointer, SEXP SOptions, SEXP SOtherPointer) \
{                                                                    \
    CHECK_VALID(SPointer);                                           \
	CHECK_VALID(SOtherPointer);                                      \
                                                                     \
	Type_* obj;                                                      \
	Type_* otherobj;                                                 \
                                                                     \
	SEXP_TO_ARTP(SPointer, Type_, obj);                              \
	SEXP_TO_ARTP(SOtherPointer, Type_, otherobj);                    \
                                                                     \
	return obj->Call_(SOptions, otherobj);                           \
}

// Declaring some functions using the above macros
	
//              FUNCTION NAME        SEXP ARG     TYPE          CALL
//              ==================== ============ ============= =================================
PRINT_FUNCTION_0(cppPrint,                        aRTcomponent, Print()     )

SEXP_FUNCTION_0(cppList,                          aRTcomponent, List()      ) // ShowDbs, ...
SEXP_FUNCTION_0(cppSummary,                       aRTcomponent, Summary()   )

SEXP_FUNCTION_0(cppShowThemes,                    aRTdb,    ShowThemes()    )
SEXP_FUNCTION_0(cppShowViews,                     aRTdb,    ShowViews()     )
SEXP_FUNCTION_0(cppLShowTables,                   aRTlayer, ShowTables()    )
SEXP_FUNCTION_0(cppDShowTables,                   aRTdb,    ShowTables()    )
 
SEXP_FUNCTION_0(cppGetBox,                        aRTlayer, GetBox()        )
SEXP_FUNCTION_0(cppGetThemeGeometry,              aRTtheme, GetGeometry()   )
SEXP_FUNCTION_0(cppGetLayerGeometry,              aRTlayer, GetGeometry()   )
SEXP_FUNCTION_0(cppGetThemeData,                  aRTtheme, GetData()       )
SEXP_FUNCTION_0(cppGetLayerData,                  aRTlayer, GetData()       )
SEXP_FUNCTION_0(cppGetTableData,                  aRTtable, GetData()       )
SEXP_FUNCTION_0(cppGetThemeAttributes,            aRTtheme, GetAttributes() )
SEXP_FUNCTION_0(cppGetLayerAttributes,            aRTlayer, GetAttributes() )
SEXP_FUNCTION_0(cppGetRaster,                     aRTlayer, GetRaster()     )
SEXP_FUNCTION_0(cppGetNext,                       aRTproxy, GetNext()       )
SEXP_FUNCTION_0(cppGetProj,                       aRTlayer, GetProj()       )
SEXP_FUNCTION_0(cppGetVisual,                     aRTtheme, GetVisual() )

VOID_FUNCTION_1(cppSetVisual,        visual,      aRTtheme, SetVisual(visual)                   )
VOID_FUNCTION_1(cppAddPoints,        points,      aRTlayer, AddPoints(points)                   )
VOID_FUNCTION_1(cppAddPolygons,      polygons,    aRTlayer, AddPolygons(polygons)               )
VOID_FUNCTION_1(cppAddLines,         lines,       aRTlayer, AddLines(lines)                     )
VOID_FUNCTION_1(cppAddShape,         shapedata,   aRTlayer, AddShape(shapedata)                 )
VOID_FUNCTION_1(cppSetAR,            restriction, aRTtheme, SetAR(SEXP_TO_STRING(restriction))  )
VOID_FUNCTION_1(cppSetSR,            restriction, aRTtheme, SetSR(SEXP_TO_STRING(restriction))  )
VOID_FUNCTION_1(cppSetTR,            restriction, aRTtheme, SetTR(SEXP_TO_STRING(restriction))  )
VOID_FUNCTION_1(cppCreateViewFromDb, viewname,    aRTdb,    CreateView(SEXP_TO_STRING(viewname)))
VOID_FUNCTION_1(cppAddRaster,        raster,      aRTlayer, AddRaster(raster)                   )
//VOID_FUNCTION_1(cppAddColumn,        data,        aRTtable, AddColumn(data)                     )
VOID_FUNCTION_1(cppCreateColumn,     data,        aRTtable, CreateColumn(data)                  )
VOID_FUNCTION_1(cppAddRows,          data,        aRTtable, AddRows(data)                       )
VOID_FUNCTION_1(cppUpdateColumns,    data,        aRTtable, UpdateColumns(data)                 )
VOID_FUNCTION_1(cppAddUser,          userName,    aRT,      AddUser(SEXP_TO_STRING(userName))   )
VOID_FUNCTION_1(cppCreateRelation,   data,        aRTtable, CreateRelation(data)                )
VOID_FUNCTION_1(cppSetVisible,       visible,     aRTtheme, SetVisible(visible)                 )

		
SEXP_FUNCTION_1(cppGetMetric,        options,     aRTlayer, GetMetric(options)                  )
SEXP_FUNCTION_1(cppGetSetMetric,     options,     aRTlayer, GetSetMetric(options)               )
		
SEXP_FUNCTION_2P(cppGetDistance,         aRTlayer, GetDistance)
SEXP_FUNCTION_2P(cppGetRelation,         aRTlayer, GetRelation)
SEXP_FUNCTION_2P(cppGetNearestNeighbors, aRTlayer, GetNearestNeighbors)

SEXP cppPP(SEXP value)
{
	printAttrib(value);
    printSEXP(value);
	return R_NilValue;
}

VOID_FUNCTION_1( cppDeleteLayer,  layername,   aRTdb,        DeleteLayer(SEXP_TO_STRING(layername)))
VOID_FUNCTION_1( cppDeleteView,   viewname,    aRTdb,        DeleteView( SEXP_TO_STRING(viewname)) )
VOID_FUNCTION_1( cppDeleteTheme,  themename,   aRTdb,        DeleteTheme(SEXP_TO_STRING(themename)))
VOID_FUNCTION_1( cppDeleteTable,  tablename,   aRTdlcommon,  DeleteTable(SEXP_TO_STRING(tablename)))
VOID_FUNCTION_1( cppDeleteDb,     dbname,      aRT,          DeleteDb(   SEXP_TO_STRING(dbname))   )

SEXP cppRemove(SEXP SPointer)
{
	CHECK_VALID(SPointer);
	
	aRTcomponent* obj;
	SEXP_TO_ARTP(SPointer, aRTcomponent, obj);
	delete obj;
	return R_NilValue;
}

SEXP cppSilent(SEXP silent)
{
	if(silent != R_NilValue)
	{
		aRTcomponent::Silent = SEXP_TO_INTEGER(silent);
		return silent;
	}

	SEXP sx = allocVector(LGLSXP, 1);
	LOGICAL(sx)[0] = aRTcomponent::Silent;
	return sx;
}

SEXP cppConfirm()
{
	char c[100];

	do
	{
		Rprintf("It is a dangerous operation. Confirm? [y/n]");
		scanf("%s", c);
	}while(c[0] != 'y' && c[0] != 'n');

	SEXP ch = allocVector(STRSXP, 1);
	char cc[2];
	cc[0] = c[0];
	cc[1] = '\0';
    SET_STRING_ELT(ch, 0, mkChar(cc));
	return ch;
}

// create a new aRT object, open a connection with a SGBD and return its pointer
SEXP cppNewaRT(SEXP user, SEXP password, SEXP port, SEXP host, SEXP dbms)
{
	SEXP pointer;
	// Database authentication parameters
	string strUser       = SEXP_TO_STRING(user);
	string strPasswd     = SEXP_TO_STRING(password);
	string strDbms       = SEXP_TO_STRING(dbms);
	port                 = AS_NUMERIC(port); // vector of double
	unsigned int intPort = (unsigned int)NUMERIC_POINTER(port)[0];
	string strHost       = SEXP_TO_STRING(host);

	ARTP_TO_SEXP( new aRT(strUser, strPasswd, intPort, strHost, strDbms), pointer )
	return pointer;
}

#define POINTER_FUNCTION(FunctionName_, Parent_, Type_, Call_) \
SEXP FunctionName_(SEXP SPointer, SEXP SData)                  \
{                                                              \
	CHECK_VALID(SPointer);                                     \
                                                               \
	Parent_* obj;                                              \
	Type_ *result;                                             \
    SEXP pointer;                                              \
	SEXP_TO_ARTP(SPointer, Parent_, obj)                       \
	result = obj -> Call_(SData);                              \
	ARTP_TO_SEXP(result, pointer);                             \
	return pointer;                                            \
}	

POINTER_FUNCTION(cppOpen,   aRTcomponent, aRTcomponent, Open     )
POINTER_FUNCTION(cppProxy,  aRTlayer,     aRTcomponent, OpenProxy)
POINTER_FUNCTION(cppLTable, aRTlayer,     aRTcomponent, OpenTable)
POINTER_FUNCTION(cppDTable, aRTdb,        aRTcomponent, OpenTable)

// POINTER_FUNCTION(cppTable, aRTdlcommon,    aRTcomponent, OpenTable)
// see why this don't work in aRTlayer.h

SEXP cppImportShape(SEXP SPointer, SEXP SLayerName, SEXP SFileName, SEXP Sproj)
{
    CHECK_VALID(SPointer);

    aRTdb* objDb;
    aRTcomponent* objComp;
    SEXP_TO_ARTP(SPointer, aRTdb, objDb)
    SEXP pointer;

    string layerName = SEXP_TO_STRING(SLayerName);
    string fileName  = SEXP_TO_STRING(SFileName);

    TeProjection* proj = NULL;
    if (!isNull(Sproj))
	{
        string strDatum = CHAR(STRING_ELT(GET_SLOT(Sproj, install("datum")),0));
        string class_   = CHAR(STRING_ELT(GET_CLASS(Sproj),0));
        TeDatum datum   = TeDatumFactory::make(strDatum);

        if(class_ == "aRTlatlong")
            proj = new TeLatLong(datum);
    }
    else
        proj = new TeNoProjection();

    objComp = objDb -> ImportShape(fileName, layerName, proj);
    ARTP_TO_SEXP(objComp, pointer);

    return pointer;
}

}

