/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

#include "aRTtheme.h"
#include "aRTdb.h"
#include "SEXPutils.h"

#include <TeRaster.h>
#include <TeInitRasterDecoders.h>

#include <sstream>

extern "C"
{

aRTtheme::aRTtheme(TeDatabase* database, string name)
{
	Database  = database;
	ThemeName = name;
}

aRTtheme::~aRTtheme()
{
	PrintSilent("Deleting aRTtheme \"%s\" from memory\n", ThemeName.c_str());
	Clear();
}

aRTcomponent* aRTtheme::Open(SEXP data)
{
	cout << "open" << endl;
	TeQuerierParams params;//(false, true);
	string table = "events";
	string col = "id";
	params.setParams(Theme(), TeSECOND);//, table, col, TeNOSTATISTIC);
    aRTproxy* p = new aRTproxy(params);
	cout << "fim open" << endl;
	return p;
}

// function to persist a theme in the database
void aRTtheme::Save(string viewname)
{
	TeLayer* layer     = Theme() -> layer();
	TeProjection* proj = layer -> projection();
	const string user  = layer -> database() -> user();
	TeDatabase* te_db  = layer -> database();
	
	TeView* view = TeUtils::GetViewFromDatabase(te_db, viewname);
	if (!view)
		error("There is no view named \'%s\' in the database!"
              "Use \'createView\' first.\n", viewname.c_str());

	if( TeUtils::GetThemeFromView(view, ThemeName) )
	{
		stringstream s;
   		s << "There is already a theme " + ThemeName +
             " into the view "  + view -> name() << endl;
		error( StreamToChar(s) );
	}

	view -> projection(proj);
	view -> add(Theme());
    Theme() -> visibleRep(TeRASTER | 0x40000000);  // raster and legend visible

	PrintSilent("Saving...\n");
	if ( !Theme() -> save() )
	{
		stringstream s;
		s << "Error saving theme: " << te_db -> errorMessage() << endl;
		error( StreamToChar(s) );
	}

	PrintSilent("Building collection...\n");
	if ( !Theme() -> buildCollection() )
	{
		stringstream s;
		s << "Error building theme: " << te_db -> errorMessage() << endl;
		error( StreamToChar(s) );
	}

	PrintSilent("Saved successfully\n");
}

enum aRTthemeSenum
{
    aRTthemeSClass = 0,
    aRTthemeSName,
	aRTthemeSLayer,
	aRTthemeSView,
    aRTthemeSTables,
    aRTthemeSAR,
    aRTthemeSTR,
    aRTthemeSSR,
    aRTthemeSSum
};

SEXP aRTtheme::Summary()
{
	SEXP info, names;

    info = allocVector(VECSXP, aRTthemeSSum); // List containing the connection attributes
    names = allocVector(STRSXP, aRTthemeSSum); // Names Vector

	string layername = Theme()->layer()? Theme()->layer()->name(): "NOT";
	
	STRING_LIST_ITEM(info, names, aRTthemeSClass, "class",          Class());
    STRING_LIST_ITEM(info, names, aRTthemeSName,  "theme",          ThemeName);
	STRING_LIST_ITEM(info, names, aRTthemeSLayer, "layer",          layername);
	STRING_LIST_ITEM(info, names, aRTthemeSView,  "view",           Theme()->layer()->database()->viewMap()[Theme()->view()]->name());
	STRING_LIST_ITEM(info, names, aRTthemeSAR,    "attribute_rest", Theme()->attributeRest());
    STRING_LIST_ITEM(info, names, aRTthemeSTR,    "temporal_rest",  Theme()->temporalRest());
    STRING_LIST_ITEM(info, names, aRTthemeSSR,    "spatial_rest",   Theme()->spatialRest());

    SET_STRING_ELT(names, aRTthemeSTables,  mkChar("tables"));

    TeAttrTableVector attrs;
    TeAttributeList  attrlist;
    SEXP s_attrs, s_names, s_attr_list;
    Theme()->getAttTables (attrs);

    s_attrs = allocVector(VECSXP, attrs.size());
    s_names = allocVector(STRSXP, attrs.size());

    for(unsigned i = 0; i < attrs.size(); i++)
	{
        SET_STRING_ELT(s_names, i, mkChar(attrs[i].name().c_str()));
        attrlist = attrs[i].attributeList();
        s_attr_list = allocVector(STRSXP, attrlist.size());

        for(unsigned j = 0; j < attrlist.size(); j++)
        {
            SET_STRING_ELT(s_attr_list, j, mkChar(attrlist[j].rep_.name_.c_str()));
        }
        SET_VECTOR_ELT(s_attrs, i, s_attr_list);
    }
    setAttrib(s_attrs, R_NamesSymbol, s_names);

    SET_VECTOR_ELT(info, aRTthemeSTables, s_attrs);
    setAttrib(info, R_NamesSymbol, names); // Set the names of the list

    return info;
}

const string aRTtheme::Print()
{
    stringstream s;

    TeAttrTableVector attrs;
    TeAttributeList  attrlist;
    Theme()->getAttTables (attrs);

	s <<  artOBJECT("aRTtheme") << "\n\n";

	s << "Theme: \""     << ThemeName;
	if(Theme()->layer())
		s  << "\"\nLayer: \"" << Theme()->layer()->name();

	s  << "\"\nTables: ";

    for(unsigned i = 0; i < attrs.size(); i++)
	{
		s <<  "\n         \"" <<  attrs[i].name() <<  "\"\n         Attributes: ";
    	attrlist = attrs[i].attributeList();

    	for(unsigned j = 0; j < attrlist.size(); j++)
		{
			s <<  "\"" << attrlist[j].rep_.name_ << "\"";
			if(j + 1 < attrlist.size()) s << ", ";
            if(!((j+1)%5)) s << "\n                      ";
        }
        s << "\n";
    }

	s << "\nAttribute Restriction: \""  << Theme()->attributeRest()
	  << "\"\nTemporal Restriction: \"" << Theme()->temporalRest()
	  << "\"\nSpatial Restriction: \""  << Theme()->spatialRest()
	  << "\"\n";

    return s.str();
}

SEXP aRTtheme::GetBox()        { return SEXPutils::GetBoxTeBox( Theme()->boxRestriction() ); }
SEXP aRTtheme::GetData()       { return SEXPutils::GetThemeData( Theme() ); }
SEXP aRTtheme::GetAttributes() { return SEXPutils::GetThemeAttributes( Theme() ); }
SEXP aRTtheme::GetGeometry()   { return SEXPutils::GetThemeGeometry( Theme() ); }

SEXP aRTtheme::GetRaster()
{ 
/*	if( !(Theme->layer()->geomRep() & TeRASTER) ) error("Theme does not have raster");
	TeRaster* raster = Theme->layer()->raster();

	TeRDecoder* decoder = new TeRDecoder( raster->decoder() );*/
	return R_NilValue;
}
		
bool aRTtheme::QueryValid()
{
    string sel;
    sel =   "SELECT count(*) " ;
    sel +=  Theme()->sqlGridFrom() + " WHERE " + Theme()->sqlWhereRestrictions();
    //cout << sel << endl;

    TeDatabasePortal *portal = Database->getPortal();
    return portal->query(sel);
}

void aRTtheme::SetAR(string restriction)
{
	PrintSilent("false\n");
	/*
	string old_rest = Theme()->attributeRest();
    Theme() -> attributeRest(restriction);
    stringstream sel, error;
    sel << "SELECT count(*) " << Theme()->sqlGridFrom() << " WHERE " << Theme()->sqlWhereRestrictions();

	TeDatabasePortal *portal = Database->getPortal();
    if ( !portal->query( sel.str() ) )
	{
       	error << "ERROR: " << portal->errorMessage() << endl;
		Theme()->attributeRest(old_rest);
	PrintSilent(StreamToChar(error));
	}

    PrintSilent("Restrictions changed\n");
	
	if(!Theme()->buildCollection()) cout << "not build!" << endl;
//	if(!Theme->save())            cout << "not saved!" << endl;
	//if(!Theme->buildCollection()) cout << "not build!" << endl;
	PrintSilent(StreamToChar(error));*/
}

void aRTtheme::SetTR(string restriction)
{
	PrintSilent("false\n");
}

void aRTtheme::SetSR(string restriction)
{
	PrintSilent("false\n");
}

SEXP aRTtheme::GetVisual()
{
	return R_NilValue;//Theme()->get
}

void aRTtheme::SetVisible(SEXP visible)
{
	bool vis = SEXP_TO_INTEGER(visible);
	Theme()->visibility(vis);
	Database->updateTheme(Theme());
}

void aRTtheme::SetVisual(SEXP options)
{
	SEXP svisual = GetListElement    (options, "visual");
	SEXP satt    = GetListElement    (options, "attribute");
	string mode  = GET_STRING_ELEMENT(options, "mode");
	
	TeGeomRep geomrep;

	if( Theme()->layer()->hasGeometry(TePOLYGONS) ) geomrep = TePOLYGONS;
	if( Theme()->layer()->hasGeometry(TeLINES) )    geomrep = TeLINES;
	if( Theme()->layer()->hasGeometry(TePOINTS) )   geomrep = TePOINTS;
	if( Theme()->layer()->hasGeometry(TeRASTER) )   geomrep = TeRASTER;

	SEXP scolor  = GetListElement(svisual, "color");
	vector<TeColor> colors = SEXPToColors(scolor);
	TeVisual visual = GetTeVisual(svisual, geomrep);

	if(colors.size() == 1) // no slicing
	{
		    Theme()->setVisualDefault (visual, geomrep);
			Database->updateTheme( Theme() );
			return;
	}		

	Theme()->resetGrouping(); // clear any grouping previously created

    TeGrouping grouping; // defines a particular way of grouping objects
	if(satt != R_NilValue)
	{	
    	TeAttributeRep rep;
    	rep.name_ = CHAR(STRING_ELT(satt, 0));
    	grouping.groupAttribute_ = rep;
	}

	grouping.groupPrecision_ = 4; // TODO: always 4?

	switch(mode[0])
	{
		case 'e': grouping.groupMode_ = TeEqualSteps;    break;
		case 'q': grouping.groupMode_ = TeQuantil;       break;
		case 's': grouping.groupMode_ = TeStdDeviation;  break;
		case 'u': grouping.groupMode_ = TeUniqueValue;   break;
		case 'r': grouping.groupMode_ = TeRasterSlicing; break;
		default:
			error("Invalid grouping mode\n");
	}
	
    grouping.groupNumSlices_ = colors.size();

	if (!Theme()->buildGrouping (&grouping)) // generate groups based on grouping definition
    {
        cout << "Error generating groups: " << Database->errorMessage() << endl;
        Database->close();
        return;
    }

	for(int i = 0; i != colors.size(); i++)
	{
		visual.color(colors[i]);
    	Theme()->setGroupingVisual(i+1,visual,TePOLYGONS);
	}
        
    if (!Theme()->saveGrouping() )
    {
        cout << "Error materializing grouping: " << Database->errorMessage() << endl;
        Database->close();
    return;
    }
}

}

