/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

#include "aRTtable.h"
#include "SEXPutils.h"

using namespace std;

extern "C"
{

aRTtable::aRTtable(TeDatabase* database, string layername, string tablename)
{
	TableName = tablename;
	LayerName = layername;
	Database  = database;
}

aRTtable::~aRTtable()
{
	PrintSilent("Deleting aRTtable object from memory\n");
	Clear();
}
	
// this is the order to appear inside R. S is from Summary
enum aRTtableSenum
{
	aRTtableSClass = 0,
	aRTtableSName,
	aRTtableSType,
	aRTtableSRows,
	aRTtableSColumns,
	aRTtableSKeys,
	aRTtableSSize
};

SEXP aRTtable::Summary()
{
    SEXP info, names, skeys, scolumns;

	TeTable table;
	vector<string> stratts;

	Database->loadTable(TableName, table);
	Database->loadTableInfo(table);
	table.primaryKeys(stratts);
	TeAttributeList attlist = table.attributeList();

	info     = allocVector(VECSXP, aRTtableSSize);  // List containing the connection attributes
    names    = allocVector(STRSXP, aRTtableSSize);  // Names Vector
	skeys    = allocVector(STRSXP, stratts.size()); // keys vector
	scolumns = allocVector(STRSXP, attlist.size());

	for(int i = 0; i < attlist.size();i++)
		SET_STRING_ELT(scolumns, i, mkChar(attlist[i].rep_.name_.c_str()));
	
	for(int i = 0; i != stratts.size(); i++)
		SET_STRING_ELT(skeys, i, mkChar(stratts[i].c_str()));
	
	STRING_LIST_ITEM (info, names, aRTtableSClass,   "class",   Class());
	STRING_LIST_ITEM (info, names, aRTtableSName,    "name",    TableName);
	STRING_LIST_ITEM (info, names, aRTtableSType,    "type",    AttrTableType(table.tableType()));
	INTEGER_LIST_ITEM(info, names, aRTtableSRows,    "rows",    table.size());
	SEXP_LIST_ITEM   (info, names, aRTtableSKeys,    "keys",    skeys);
	SEXP_LIST_ITEM   (info, names, aRTtableSColumns, "columns", scolumns);
	
    setAttrib(info, R_NamesSymbol, names); // Set the names of the list
	
	return info;
}

const string aRTtable::Print()
{
	stringstream s;

	if (!IsValid()) error("Invalid object!");

    TeTable table;
	table.name(TableName);
	Database->loadTable(TableName, table);
	Database->loadTableInfo(table);
	s << artOBJECT("aRTtable")  << "\n\n"
	  << "Table: \"" << TableName << "\"" << endl
	  << "Type: " << AttrTableType(table.tableType()) << endl;
	if(LayerName != "")
		s << "Layer: " << LayerName << endl;
		
	s << "Rows: " << table.size() << endl
	  << "Attributes: " << endl; 

	vector<string> stratts;
	
	TeAttributeList attlist = table.attributeList();
	
	for(int i = 0; i < attlist.size();i++)
	{
		s << "    " << attlist[i].rep_.name_ << ": "
		  << AttrDataType(attlist[i].rep_.type_);

		if(attlist[i].rep_.type_ == TeSTRING)
			s << "[" << attlist[i].rep_.numChar_ << "]";
		
		if(attlist[i].rep_.isPrimaryKey_) s << " (key)";

		s << endl;
	}

	return s.str();
}

// addRows needs all attributes, and calculates the order according
// to terralib attributes
void aRTtable::AddRows(SEXP data)
{
	TeTable table(TableName);

	Database->loadTableInfo(table);

	SEXP colnames = GetListElement     (data, "colnames");
	SEXP matrix   = GetListElement     (data, "matrix");
	int lrows     = GET_INTEGER_ELEMENT(data, "length");

	int lcols = length(colnames);
	
	TeAttributeList attlist = table.attributeList();
	int order[attlist.size()];

	for(int i = 0; i != attlist.size(); i++)
	{
		order[i] = -2;
		
		if(attlist[i].rep_.type_==TeDATETIME)
		{
			attlist[i].dateTimeFormat_ = "YYYYsMMsDDsHHsmmsSS";
			attlist[i].dateSeparator_ = "-";
			attlist[i].timeSeparator_ = ":";
			attlist[i].dateChronon_ = TeSECOND;
			attlist[i].indicatorPM_ = "";
			attlist[i].indicatorAM_ = "";
		}
		
		// check for table attribute in colnames
		for(int j = 0; j < lcols; j++)
		{						
			string value = CHAR(STRING_ELT(colnames,j));

			if( attlist[i].rep_.name_ == value)
				order[i] = j;
		}

		if(order[i] == -2)
			error("Attribute \'%s\' not found\n", attlist[i].rep_.name_.c_str());
	}

	// changing datetime format (the format is not stored in the database)
	table.setAttributeList(attlist);

	if(attlist.size() != lcols)
		error("There are more attributes in data.frame than in the table\n");

	for(int i = 0; i != lrows; i++)
	{
		TeTableRow row;

		for(int j = 0; j != attlist.size(); j++)
		{
			row.push_back( CHAR(STRING_ELT(matrix, i+order[j]*lrows)) );
		}
		table.add(row);
	}	
	
	if( !Database->insertTable (table) )
		error("Could not insert data in the table\n");

	PrintSilent("Data inserted successfully\n");
}

void aRTtable::CreateColumn(SEXP data)
{
	TeAttributeRep atrep;
	
    int size            = GET_INTEGER_ELEMENT(data, "length");
    string type         = GET_STRING_ELEMENT (data, "type");
    
	atrep.name_         = GET_STRING_ELEMENT (data, "colname");
    atrep.decimals_     = GET_INTEGER_ELEMENT(data, "decimals");
	atrep.isPrimaryKey_ = GET_BOOL_ELEMENT   (data, "key");
	atrep.isAutoNumber_ = GET_BOOL_ELEMENT   (data, "autonumber");

	TeTable table(TableName);

	if( !Database->loadTableInfo(table) )
        error("Could not load the table info!\n");

	TeAttribute att;
	if( Database->columnExist(TableName, atrep.name_, att))
		error("Column \'%s\' already exists in \'%s\'\n", atrep.name_.c_str(), TableName.c_str());
	
    switch( type[0] )
    {
        case 'i': // integer
            atrep.type_ = TeINT;
            break;
		case 'f': case 'c':// factor, character
            atrep.type_ = TeSTRING;
            atrep.numChar_ = size;
            break;
        case 'n': // numeric
            atrep.type_ = TeREAL;
            break;
        default:
            error("Invalid attribute type: %s\n", type.c_str());
	}

	Database->addColumn(TableName, atrep);
	PrintSilent("Column \'%s\' created in \'%s\'\n", atrep.name_.c_str(), TableName.c_str());
}

// update columns doesn't need all atributes, and data
// doesn't need to be in the same order as the database table
void aRTtable::UpdateColumns (SEXP data)
{
	TeTable table(TableName);
	
	Database->loadTableInfo(table);
	SEXP colnames = GetListElement     (data, "colnames");
	SEXP matrix   = GetListElement     (data, "matrix");
	int lrows     = GET_INTEGER_ELEMENT(data, "length");

	int lcols = length(colnames);
	bool id_in_rownames = false;
	
	TeAttributeList attlist = table.attributeList();
	TeAttributeList attlist_result;
	
	for(int i = 0; i != lcols; i++)
	{
		for(int j = 0; j != attlist.size(); j++)
			if(attlist[j].rep_.name_ == string(CHAR(STRING_ELT(colnames,i))))
				attlist_result.push_back(attlist[j]);
	}

	table.setAttributeList(attlist_result);
	
	for(int i = 0; i != lrows; i++)
	{
		TeTableRow row;

		for(int j = 0; j != lcols; j++)
			row.push_back( CHAR(STRING_ELT(matrix, i+j*lrows)) );
		
		table.add(row);
	}	
	
	if( !Database->updateTable (table) )
		error("Could not update the table\n");

	PrintSilent("Columns updated successfully\n");
}

void aRTtable::CreateRelation(SEXP data)
{
	string fieldName     = GET_STRING_ELEMENT(data, "attr");
	string relatedTable  = GET_STRING_ELEMENT(data, "rtable");
	string relatedField  = GET_STRING_ELEMENT(data, "rattr");

	TeTable table(TableName);
	Database->loadTableInfo(table);

	int relId;
	if( Database->insertRelationInfo (table.id(), fieldName, relatedTable, relatedField, relId) )
		PrintSilent("Relation created successfully\n");
	else
		error("Could not create the relation\n");
}

SEXP aRTtable::GetData()
{
    TeTable table;
	table.name(TableName);
	Database->loadTable(TableName, table);
	Database->loadTableInfo(table);

	SEXP result;
	SEXP *each_column;
	SEXP colnames;
	SEXP rownames;
	
	TeAttributeList attlist = table.attributeList();
	
	each_column = new SEXP[attlist.size()];
	colnames    = PROTECT(allocVector( STRSXP, attlist.size() ));
	result      = PROTECT(allocVector( VECSXP, attlist.size() ));
	rownames    = PROTECT(allocVector( STRSXP, table.size() ));
	
	for(int i = 0; i != attlist.size(); i++)
	{
		each_column[i] = PROTECT(allocVector( STRSXP, table.size() ));
		SET_STRING_ELT(colnames, i, mkChar(attlist[i].rep_.name_.c_str()));
	}
	
	for(int i = 0; i < table.size(); i++)
	{
		for(int j = 0; j < attlist.size(); j++)
		{
			string value = table(i,j);
//			if( attlist[j].rep_.type_ == TeDATETIME )
//			{
//				TeTime time(value, TeSECOND, "DDsMMsYYYYsHHsmmsSS");
//				value = time.getDateTime("YYYYsMMsDDsHHsmmsSS", ":", ":","","");				
//			}
			SET_STRING_ELT( each_column[j], i, mkChar(value.c_str()) );
		}
		
		stringstream str;
		str << i+1; // rownames must start from 1
		SET_STRING_ELT(rownames, i, mkChar(str.str().c_str()));
	}

	setAttrib(result, R_NamesSymbol, colnames);
	
	for(int i = 0; i != attlist.size(); i++)
		SET_VECTOR_ELT(result, i, each_column[i]);

	result = AsDataFrame(result, rownames);
	UNPROTECT(attlist.size()+3);
	delete[] each_column;

	return result;
}

void aRTtable::CreateLinkIds(string idname)
// creates an empty table and add rows with the object's ids
{
	TeAttributeList attList;
	TeAttribute at;
    at.rep_.name_ = "tempCol";
    at.rep_.type_ = TeSTRING;
    at.rep_.numChar_ = 100;
    at.rep_.isAutoNumber_ = false;
    at.rep_.isPrimaryKey_ = false;
    attList.push_back(at);

    if(Database->tableExist("tempTable"))
        if(Database->execute("DROP TABLE tempTable") == false)
			error("Fail to drop the temporary table");

    if(Database->createTable("tempTable", attList) == false)
		error("Fail to create the temporary table");

    string popule, geo;
    if(Layer()->hasGeometry(TePOLYGONS))
    {
        geo = Layer()->tableName(TePOLYGONS);
        popule = "INSERT INTO tempTable (tempCol) SELECT object_id FROM " + geo;
        if(Database->execute(popule) == false)
			error("Fail to drop the temporary table");
    }
    if(Layer()->hasGeometry(TePOINTS))
    {
        geo = Layer()->tableName(TePOINTS);
        popule = "INSERT INTO tempTable (tempCol) SELECT object_id FROM " + geo;
        if(Layer()->database()->execute(popule) == false)
			error("Fail to drop the temporary table");
    }
    if(Layer()->hasGeometry(TeLINES))
    {
        geo = Layer()->tableName(TeLINES);
        popule = "INSERT INTO tempTable (tempCol) SELECT object_id FROM " + geo;
        if(Layer()->database()->execute(popule) == false)
			error("Fail to drop the temporary table");
    }

	popule = "INSERT INTO " + TableName + " (" + idname + ") SELECT DISTINCT tempCol FROM tempTable";
    if(Database->execute(popule) == false)
		error("Fail to insert the objects in the table");

    PrintSilent("Link ids added successfully\n");
}

bool aRTtable::IsValid()
{
	return Valid;
}

}

