/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

#include "aRTlayer.h"

#include "aRTstatictable.h"
#include "aRTmediatable.h"
#include "aRTeventtable.h"
#include "aRTdynattrtable.h"

#include "SEXPbasic.h"
#include "SEXPutils.h"

#include <TeProjection.h>
#include <TeImportRaster.h>
#include <TeDecoderMemoryMap.h>
#include <TeInitRasterDecoders.h>
#include <TeRaster.h>
#include <TeDatabaseUtils.h>

#include <stdio.h>
#include <iostream>

using namespace std;

SEXP aRTlayer::GetProj()
{
	string sproj = TeGetSProjFromTeProjection(Layer()->projection());

	SEXP resp;

	resp  = allocVector(STRSXP, 1);
	SET_STRING_ELT( resp, 0, mkChar(sproj.c_str()) );
	return resp;
}

SEXP aRTlayer::GetBox()
{
	TeRepresentation* rep;
	rep = Layer()->getRepresentation( GeomRep() );
	
	if(rep) return SEXPutils::GetBoxTeBox(rep->box_);
	return R_NilValue;
}

TeGeomRep aRTlayer::GeomRep()
{
	if( Layer()->hasGeometry(TePOLYGONS) ) return TePOLYGONS;
	if( Layer()->hasGeometry(TeLINES) )    return TeLINES;
	if( Layer()->hasGeometry(TePOINTS) )   return TePOINTS;
	
	error("Layer does not have a valid geometry\n");
	return TeGEOMETRYNONE;
}

SEXP aRTlayer::GetNearestNeighbors(SEXP options, aRTlayer* alayer)
{
	string objId = GET_STRING_ELEMENT(options, "id");
	int numRes   = GET_INTEGER_ELEMENT(options, "quantity");

	SEXP result;	
	TeGeomRep visRep = GeomRep();
	TeGeomRep actRep = alayer->GeomRep();
	string actGeomTable;
	string visGeomTable;
	visGeomTable = Layer()->tableName(visRep);
	actGeomTable = alayer->Layer()->tableName(actRep);

	string visCollTable;
	Keys visIdsOut;

	Database->nearestNeighbors(actGeomTable,
							   actRep,
							   objId,
							   visGeomTable,
							   visCollTable,
							   visRep,
							   visIdsOut,
							   numRes);
	
    PROTECT(result = allocVector(STRSXP, visIdsOut.size()));

    for(unsigned i = 0; i < visIdsOut.size(); i++)
        SET_STRING_ELT(result, i, mkChar(visIdsOut[i].c_str()));

    UNPROTECT(1);
    return result;
}

SEXP aRTlayer::GetDistance(SEXP ids, aRTlayer* alayer)
{
	SEXP result;

	double distance;
	
	TeGeomRep visRep = GeomRep();
	TeGeomRep actRep = alayer->GeomRep();
	string actGeomTable;
	string visGeomTable;
	visGeomTable = Layer()->tableName(visRep);
	actGeomTable = alayer->Layer()->tableName(actRep);

	string objId1 = CHAR(STRING_ELT(ids,0));
	string objId2 = CHAR(STRING_ELT(ids,1));

	bool right;
	right = Database->calculateDistance(actGeomTable,
					                    actRep,
										objId1,
										visGeomTable,
										visRep,
										objId2,
										distance);
   
	if(!right) error("Could not calculate the distance\n");
	PROTECT(result = allocVector(REALSXP, 1));
    REAL(result)[0] = distance;     	
	UNPROTECT(1);
	return result;
}

SEXP aRTlayer::GetRelation(SEXP options, aRTlayer* alayer)
{
	string relation = GET_STRING_ELEMENT(options, "relation");
	SEXP ids        = GetListElement(options, aRTstrId);

	if(Database != alayer->Database)
		error("The layers must be in the same database\n");

	SEXP result;
	TeGeomRep visRep = GeomRep();
	TeGeomRep actRep = alayer->GeomRep();
	string actGeomTable;
	string visGeomTable;
	visGeomTable = Layer()->tableName(visRep);
	actGeomTable = alayer->Layer()->tableName(actRep);
	int relate = TeUNDEFINEDREL;
	Keys actIdsIn, visIdsOut;
	
	switch(relation[0])
	{
		case 'c':
			switch(relation[2])
			{
				case 'n': relate = TeCONTAINS;  break;
				case 'o': relate = TeCROSSES;   break;
				case 'v': relate = relation[5] == 's'? TeCOVERS: TeCOVEREDBY;  break;
			}
			break;
		case 'o': relate = TeOVERLAPS;   break;
		case 'd': relate = TeDISJOINT;   break;
		case 't': relate = TeTOUCHES;    break;
		case 'w': relate = TeWITHIN;     break;
		case 'i': relate = TeINTERSECTS; break;
		case 'e': relate = TeEQUALS;     break;
		default:
			error("Relation %s does not exist\n", relation.c_str());
	}

	if(ids == R_NilValue)
	{
    	string sqlstr = "select object_id from " + actGeomTable;
	    TeDatabasePortal* portal = alayer->Database->getPortal();
		portal->query(sqlstr);

	    while( portal->fetchRow() )                              
	    {                                                        
	        string id = portal->getData(0);                             
	        actIdsIn.push_back(id);
		}
	}	
	else
	{
		for(int i = 0; i < LENGTH(ids); i++)
			actIdsIn.push_back( CHAR(STRING_ELT(ids,i)) );
	}
	
	bool right;

	right = Database->spatialRelation (actGeomTable,
			        		           actRep,
					        		   actIdsIn,
							           visGeomTable,
							           visRep,
							           visIdsOut,
							           relate);

	if(!right) error("Could not execute the spatial relation\n");
	
    PROTECT(result = allocVector(STRSXP, visIdsOut.size()));
	
	for(unsigned i = 0; i < visIdsOut.size(); i++)
    	SET_STRING_ELT(result, i, mkChar(visIdsOut[i].c_str()));     	

	UNPROTECT(1);
	return result;
}

SEXP aRTlayer::GetSetMetric(SEXP options)
{
	string metric = GET_STRING_ELEMENT(options, "metric");
	SEXP ids      = GetListElement(options, aRTstrId);
	string id1, id2;
	SEXP result, colnames;
	TeGeomRep actRep = GeomRep();
	string actGeomTable;
	TeGeometryVect geomvec;
	Keys keys;
	TePolygonSet ps;
	int i = 0;
	
	actGeomTable = Layer()->tableName(actRep);

	PROTECT(result   = allocVector(VECSXP, 1));
	PROTECT(colnames = allocVector(STRSXP, 1));

	bool good = false;
	switch(metric[0])
	{
		case 'u':
			for(i = 0; i < LENGTH(ids); i++)
				keys.push_back( CHAR(STRING_ELT(ids,i)) );
			
			good = Database->geomUnion(actGeomTable, actRep, keys, geomvec);
			break;
		case 'i':
			for(i = 0; i < LENGTH(ids); i++)
				keys.push_back( CHAR(STRING_ELT(ids,i)) );
			
			good = Database->geomIntersection(actGeomTable, actRep, keys, geomvec);
			break;
		case 'd':
			id1 = CHAR(STRING_ELT(ids,0));
			id2 = CHAR(STRING_ELT(ids,1));
			//cout << id1.c_str() << "  " << id2.c_str() << endl;
			good = Database->geomDifference(actGeomTable, actRep, id1, id2, geomvec);
			break;
		case 'x':
			id1 = CHAR(STRING_ELT(ids,0));
			id2 = CHAR(STRING_ELT(ids,1));
			//cout << id1.c_str() << "  " << id2.c_str() << endl;

			good = Database->geomXOr(actGeomTable, actRep, id1, id2, geomvec);
			break;
		default:
			error("Invalid set metric: %s\n", metric.c_str());
	}

	if(!good) error("Operation could not be done\n");

    TeGeometryVect::iterator it = geomvec.begin();
    while (it!= geomvec.end())
    {
        TeGeometry* geom = (*it);

		// "spatialQuery.ui.h" line 1550
		if (dynamic_cast<TePolygonSet*> (geom))
        {
			TePolygonSet polySet, *polSet;
            polSet = new TePolygonSet();
            polSet = (TePolygonSet*) geom;
            polySet = *polSet;

			SEXP other;
			
			PROTECT(other = allocVector(VECSXP,1 ));

			SET_VECTOR_ELT(other, 0, GetPolygonsTePolygonSet(polySet));
		    SET_STRING_ELT(colnames, 0, mkChar(aRTstrPolygon));       
                                                             
		    setAttrib(result, R_NamesSymbol, colnames);              
                                                             
		    SET_VECTOR_ELT(result, 0, other);
			UNPROTECT(3);
			return result;
		}
		else error("Type returned not implemented\n");

		it++;
	}

	return R_NilValue;	
}

SEXP aRTlayer::GetMetric(SEXP options)
{
	string metric = GET_STRING_ELEMENT(options, "metric");

	SEXP result, colnames, ids, other;
	TeGeomRep actRep = GeomRep();
	string actGeomTable;
	Keys keys;
	TeRepresPointerVector representations;
	TePolygonSet ps;
	TePointSet cs;
	TeBox box;
	double value;
	string id;
	int i = 0;
	
	actGeomTable = Layer()->tableName(actRep);

	string sqlstr = "select object_id from " + actGeomTable;
	TeDatabasePortal* portal = Database->getPortal();
	portal->query(sqlstr);

	PROTECT(result   = allocVector(VECSXP, 2));
	PROTECT(colnames = allocVector(STRSXP, 2));
	PROTECT(ids = allocVector(STRSXP, portal->numRows()));

#define METRIC_FUNCTION(type_, metric_call_, set_call_, attrib_name_ , clear_call_)\
    PROTECT(other = allocVector(type_, portal->numRows()));  \
    while( portal->fetchRow() )                              \
    {                                                        \
        id = portal->getData(0);                             \
        keys.clear();                                        \
        keys.push_back(id);                                  \
        clear_call_;                                         \
        Database->metric_call_;                              \
        set_call_;                                           \
        SET_STRING_ELT(ids, i, mkChar(id.c_str()));          \
        i++;                                                 \
    }                                                        \
    SET_STRING_ELT(colnames, 0, mkChar(aRTstrId));           \
    SET_STRING_ELT(colnames, 1, mkChar(attrib_name_));       \
                                                             \
    setAttrib(result, R_NamesSymbol, colnames);              \
                                                             \
	SET_VECTOR_ELT(result, 0, ids);                          \
    SET_VECTOR_ELT(result, 1, other);                        \
    UNPROTECT(4);                                            \
    return result;

	switch(metric[0])
	{
		case 'a':
			METRIC_FUNCTION(REALSXP,
							calculateArea(actGeomTable, actRep, keys, value),
							REAL(other)[i] = value,
							"area",)
		case 'l':
			METRIC_FUNCTION(REALSXP,
							calculateLength(actGeomTable, actRep, keys, value),
							REAL(other)[i] = value,
							"length",)
		case 'c':
			METRIC_FUNCTION(VECSXP,
							Centroid(actGeomTable, actRep, cs, keys),
							SET_VECTOR_ELT(other, i, GetPointsTePointSet(cs)),
							aRTstrPoint,
							cs.clear())
			break;
		case 'b':
			// pega o buffer interno e externo. Lubia: apenas um dos dois?
			PrintSilent("This function can be slow\n");
			value = GET_REAL_ELEMENT(options, "dist");
			METRIC_FUNCTION(VECSXP,
							Buffer(actGeomTable, actRep, keys, ps, value),
							SET_VECTOR_ELT(other, i, GetPolygonsTePolygonSet(ps)),
							aRTstrPolygon,
							ps.clear())
		case 'h':
			METRIC_FUNCTION(VECSXP,
							ConvexHull(actGeomTable, actRep, keys, ps),
							SET_VECTOR_ELT(other, i, GetPolygonsTePolygonSet(ps)),
							aRTstrPolygon,
							ps.clear())
			break;
		case 'r':
			// pegar a coluna da tabela onde tem geometria. Lubia?
			//Database->getMBRGeom (actGeomTable, "1", box);//, string colGeom)
			return GetBoxTeBox(box);
		default:
			error("Invalid metric: %s\n", metric.c_str());
	}
	return R_NilValue;
}

SEXP aRTlayer::GetRaster()
{
    if( !(Layer()->geomRep() & TeRASTER) ) error("Layer does not have raster data");
      TeRaster* raster = Layer()->raster();

    TeDecoderSEXP* decoder = new TeDecoderSEXP( raster->decoder() );
    return decoder->data();
}

SEXP aRTlayer::GetGeometry()   { return SEXPutils::GetLayerGeometry  ( Layer() ); }
SEXP aRTlayer::GetData()       { return SEXPutils::GetLayerData      ( Layer() ); }
SEXP aRTlayer::GetAttributes() { return SEXPutils::GetLayerAttributes( Layer() ); }

void aRTlayer::AddRaster(SEXP sraster)
{
	if(!SEXPutils::AddRaster(Layer(), sraster))
		error("Could not import the raster");

	PrintSilent("Raster saved successfully\n");
	Layer()->loadLayerTables();
}

aRTlayer::aRTlayer(TeDatabase* database, string layername)
{
	Database = database;
	LayerName = layername;
}

aRTlayer::~aRTlayer()
{
	PrintSilent("Deleting aRTlayer \"%s\" from memory\n", LayerName.c_str());
	Clear();
}

void aRTlayer::AddShape(SEXP options)
{
	string tablename = GET_STRING_ELEMENT  (options, "tablename");
	string filename  = GET_STRING_ELEMENT  (options, "filename");
	string id        = GET_STRING_ELEMENT  (options, "id");
	int length       = GET_INTEGER_ELEMENT (options, "length");

	if(!TeImportShape(Layer(), filename, tablename, id, length))
		error("Shape could not be imported\n");
	
	PrintSilent("Shape imported successfully\n");
}

void aRTlayer::AddPoints(SEXP sps)
{
	if(!Layer()->geomRep() | TePOINTS)
		Layer()->addGeometry(TePOINTS);
	PrintSilent("converting...\n");
	
	TePointSet ps = SEXPutils::GetTePointSet(sps);
	PrintSilent("adding...\n");
	if( !Layer()->addPoints(ps) ) error("Points could not be added\n");
	stringstream s;

	s << ps.size() << " points were added to the layer\n";
	PrintSilent(StreamToChar(s));
	Layer()->loadLayerTables();
}

void aRTlayer::AddPolygons(SEXP polset)
{
   if(!Layer()->geomRep() | TePOLYGONS)
        Layer()->addGeometry(TePOLYGONS);
	PrintSilent("converting...\n");

	TePolygonSet ps = SEXPutils::GetTePolygonSet(polset);
	PrintSilent("adding...\n");
	if( !Layer()->addPolygons(ps) ) error("Polygons could not be added!\n");
	stringstream s;

	s << ps.size() << " polygons were added to the layer\n";
	PrintSilent(StreamToChar(s));
	Layer()->loadLayerTables();
}

void aRTlayer::AddLines(SEXP lineset)
{
   if(!Layer()->geomRep() | TeLINES)
        Layer()->addGeometry(TeLINES);
	PrintSilent("converting...\n");

	TeLineSet ls = SEXPutils::GetTeLineSet(lineset);
	PrintSilent("adding...\n");
	if( !Layer()->addLines(ls) ) error("Lines could not be added!\n");
	stringstream s;

	s << ls.size() << " lines were added to the layer\n";
	PrintSilent(StreamToChar(s));
	Layer()->loadLayerTables();
}

// this is the order to appear inside R. S is from Summary
enum aRTlayerSenum
{
	aRTlayerSClass = 0,
	aRTlayerSLayer,
	aRTlayerSPolygons,
	aRTlayerSLines,
	aRTlayerSPoints,
	aRTlayerSRaster,
	aRTlayerSProjection,
	aRTlayerSDatum,
	aRTlayerSLongitude,
	aRTlayerSLatitude,
	aRTlayerSSize
};
					   
SEXP aRTlayer::Summary()
{
        SEXP info, names;

        info    = allocVector(VECSXP, aRTlayerSSize);  // List containing the connection attributes
        names   = allocVector(STRSXP, aRTlayerSSize);  // Names Vector

		string raster = "no";
	   if( Layer()->geomRep() & TeRASTER) raster = "yes";
		
	   	STRING_LIST_ITEM(info, names, aRTlayerSRaster,     "raster",     raster);
    	STRING_LIST_ITEM(info, names, aRTlayerSClass,      "class",      Class());
	    STRING_LIST_ITEM(info, names, aRTlayerSProjection, "projection", Layer()->projection()->name());
		STRING_LIST_ITEM(info, names, aRTlayerSLayer,      "layer",      LayerName);
		STRING_LIST_ITEM(info, names, aRTlayerSDatum,      "datum",      Layer()->projection()->datum().name());
		
        DOUBLE_LIST_ITEM(info, names, aRTlayerSLongitude,  "longitude",  Layer()->projection()->lon0());
        DOUBLE_LIST_ITEM(info, names, aRTlayerSLatitude,   "latitude",   Layer()->projection()->lat0());
        DOUBLE_LIST_ITEM(info, names, aRTlayerSPolygons,   "polygons",   Layer()->nGeometries(TePOLYGONS));
        DOUBLE_LIST_ITEM(info, names, aRTlayerSPoints,     "points",     Layer()->nGeometries(TePOINTS));
        DOUBLE_LIST_ITEM(info, names, aRTlayerSLines,      "lines",      Layer()->nGeometries(TeLINES));

		setAttrib(info, R_NamesSymbol, names); // Set the names of the list

        return info;
}

const string aRTlayer::Print()
{
	stringstream s;

	if (!IsValid()) error("Invalid object!");
		 
    s << artOBJECT("aRTlayer")     << "\n\n"
	  << "Layer: \""              << LayerName                             << "\"" << endl
      << "Number of polygons: "   << Layer()->nGeometries(TePOLYGONS)      << endl
      << "Number of lines: "      << Layer()->nGeometries(TeLINES)         << endl
      << "Number of points: "     << Layer()->nGeometries(TePOINTS)        << endl;

	if(Layer()->hasGeometry(TeRASTER))	s << "Layer has raster data" << endl;
	else s << "Layer does not have raster data" << endl;
	 
	if (Layer() -> projection() == NULL)
    {
        s << "Layer has no projection!" << endl;
		return s.str();
	}

	s << "Projection Name: \""   << Layer()->projection()->name()         << "\""<< endl
	  << "Projection Datum: \""  << Layer()->projection()->datum().name() << "\""<< endl
	  << "Projection Longitude: "<< Layer()->projection()->lon0()                << endl
	  << "Projection Latitude: " << Layer()->projection()->lat0()                << endl;


	TeAttrTableVector tvector = Layer()->attrTables();

	s << "Tables:";

	if(!tvector.size()) s << " (none) ";
	else for(int i = 0; i < tvector.size(); i++)
	{
		s << "\n    \"" << tvector[i].name() << "\": ";
		s << AttrTableType(tvector[i].tableType());
	}
	s << endl;
	
	return s.str();
}

SEXP aRTlayer::ShowTables()
{
	SEXP vectorTables;
	TeAttrTableVector atts;
	TeAttributeList attrList;
	
	if (!Layer() -> getAttrTables(atts)) error("Error loading tables from layer");

	vectorTables = allocVector(STRSXP, atts.size());

	for (unsigned i = 0; i < atts.size(); i++)
		SET_STRING_ELT(vectorTables, i, mkChar(atts[i].name().c_str()));

	return vectorTables;
}

aRTcomponent* aRTlayer::OpenTable(SEXP options)
{
	string table_name = GET_STRING_ELEMENT  (options, "tablename");
	bool create       = GET_BOOL_ELEMENT    (options, "create");
	string type       = GET_STRING_ELEMENT  (options, "type");

	aRTtable* child;
	
    switch(type[0])
    {
        case 's': child = new aRTstatictable  (Database, LayerName, table_name); break;
        case 'm': child = new aRTmediatable   (Database, LayerName, table_name); break;
        case 'e': child = new aRTeventtable   (Database, LayerName, table_name); break;
        case 'a': child = new aRTdynattrtable (Database, LayerName, table_name); break;
//        case 'd': atttype = TeDynGeomDynAttr;              break;
//        case 'g': atttype = TeGeomAttrLinkTime;            break;
        case 'x': error("External table must be build from databases\n");       break;
        default:  error("Type not recognized\n");
    }

	if( !Database->tableExist(child->Name()) ) // the table name can change as in mediatable
	{
		if(!create) error("Table \'%s\' does not exist, set create=T to create one\n", table_name.c_str());
	
		PrintSilent("Creating table \'%s\'\n", table_name.c_str());
		child->Create(options);
	}
	else if(create) warning("Trying to create an existent table \'%s\'\n", table_name.c_str());
	
	Parent->AddChild(child);
	return child;
}

aRTcomponent* aRTlayer::OpenProxy(SEXP options)
{
	bool loadgeom = GET_BOOL_ELEMENT(options, "loadgeom");
	bool loadatt  = GET_BOOL_ELEMENT(options, "loadatt");

	TeQuerierParams params(loadgeom, loadatt); 
    params.setParams(Layer());
	
	aRTproxy* child =  new aRTproxy(params);
	Parent->AddChild(child);
	return child;
}

aRTcomponent* aRTlayer::Open(SEXP options)
{
	string theme_name = GET_STRING_ELEMENT(options, "themename");
	bool create       = GET_BOOL_ELEMENT  (options, "create");
	string viewname   = GET_STRING_ELEMENT(options, "viewname");
	SEXP tableNames   = GetListElement    (options, "tables");
	string attrest    = GET_STRING_ELEMENT(options, "attrest");

	if( TeUtils::GetThemeFromDatabase(Database, theme_name) )
	{
		if(create) warning("Trying to create an existent theme\n");
		PrintSilent("Opening an existent theme\n");
		TeUtils::GetThemeFromDatabase(Database, theme_name)->loadThemeTables();
		aRTtheme* child = new aRTtheme(Database, theme_name);
		Parent->AddChild(child);
		return child;
	}
	
	if(!create)
	    error("Theme \'%s\' does not exist! If you want to create set create=TRUE\n", theme_name.c_str());
		
	PrintSilent("Creating theme \'%s\'\n", theme_name.c_str());
    TeTheme* te_theme = new TeTheme( theme_name, Layer() );

	te_theme->attributeRest(attrest);
	
    // try to create a view. no problem if it already exists.
	((aRTdb*)Parent)->CreateView(viewname);

	// selecting the tables
	TeLayer* layer = Layer();
	vector<string> attrTableNames;
	TeTable aux;
	TeAttrTableVector atts;
	tableNames = AS_CHARACTER(tableNames);

    // check the tables
	for (int i = 0; i < length(tableNames); i++)
	{
	    string tableName = CHAR(STRING_ELT(tableNames, i));
	    if (!layer -> getAttrTablesByName(tableName, aux))
	    {
	        error("Table \"%s\" inexist", tableName.c_str());
	    }
	    else
	    {
	        TeTable table(tableName);
	        Database -> loadTableInfo(table);
	        atts.push_back(table);
	    }
	}
	if( !te_theme -> setAttTables (atts) )
		error("Could not set the attribute tables");
		
	TeProjection* proj = layer -> projection();
	const string user  = layer -> database() -> user();
	TeView* view = TeUtils::GetViewFromDatabase(Database, viewname);
	if (!view)
	    error("There is no view named \'%s\' in the database!"
	          "Use \'createView\' first.\n", viewname.c_str());

	// TODO: set the projection only if it is the first theme of the view
	view -> projection(proj);
	view -> add(te_theme);
	te_theme -> visibility(false);
	int allrep = layer -> geomRep();
	te_theme -> visibleRep(allrep);

	if( Layer()->hasGeometry(TeRASTER) )
	{
		te_theme->visibleRep(TeRASTER | 0x40000000);
			
		if( !te_theme->save() )
		{
			stringstream s;
	        s << "Fail to save theme \'" << theme_name << "\': " << Database->errorMessage() << endl;
	        error( StreamToChar(s) );
		}
		else PrintSilent("Theme \'%s\' saved successfully\n", theme_name.c_str());
	}
	else // not a raster theme
	{
    	if ( !te_theme -> save() )
    	{
    	    stringstream s;
    	    s << "Error saving theme \'" << theme_name << "\': " << Database->errorMessage() << endl;
    	    error( StreamToChar(s) );
    	}
		PrintSilent("Theme \'%s\' saved successfully\n", theme_name.c_str());

    	if ( !te_theme -> buildCollection() )
    	{
    	    stringstream s;
    	    s << "Error building theme \'" << theme_name << "\': " << Database->errorMessage() << endl;
    	    error( StreamToChar(s) );
    	}
    	PrintSilent("Collection of theme \'%s\' built succesfully\n", theme_name.c_str());
		
	}
	aRTtheme* child = new aRTtheme(Database, theme_name);
	Parent->AddChild(child);
	te_theme->loadThemeTables();
	return child;
}

bool aRTlayer::IsValid()
{
	if(Valid && !Layer())
	{
		Valid = false;
		error("Layer removed from the database!");
	}

	return Valid;
}

