/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

#include "aRTdb.h" 
#include "aRTexternaltable.h"

#include <sstream>
#include <TeSTElementSet.h>
#include <TeSTEFunctionsDB.h>
#include <TeInitQuerierStrategy.h>
#include <TeDatabaseUtils.h>

aRTdb::aRTdb(TeDatabase* database)
{
	Database = database;
	if(!Database -> loadLayerSet())
	{
		delete this;
		error("Could not load the layer set!\n");
	}
	if(!Database -> loadViewSet(Database -> user()))
	{
		delete this;
		error("Could not load the view set!\n");
	}
}

aRTdb::~aRTdb()
{
	PrintSilent("Deleting aRTdb \"%s\" from memory\n", Database->databaseName().c_str());
	delete Database;
	Clear();
}

SEXP aRTdb::ShowViews()
{
	SEXP vectorViews;

	TeViewMap& view_map = Database -> viewMap();
	TeView* view;
	TeViewMap::iterator it;
	
	vectorViews = view_map.size() > 0 ? allocVector(STRSXP, view_map.size()) : R_NilValue;
	
	int i = 0;
	for (it = view_map.begin(); it != view_map.end(); ++it)
	{
		view = it -> second;
		SET_STRING_ELT(vectorViews, i, mkChar(view -> name().c_str()));
		i++;
    }

	return vectorViews;
}

SEXP aRTdb::ShowThemes()
{
	SEXP vectorThemes;

	TeThemeMap& theme_map = Database -> themeMap();
	TeTheme* theme;
	TeThemeMap::iterator it;
	vectorThemes = theme_map.size() > 0 ? allocVector(STRSXP, theme_map.size()) : R_NilValue;
	int i = 0;
	for (it = theme_map.begin(); it != theme_map.end(); ++it)
	{
		theme = it -> second;
		SET_STRING_ELT(vectorThemes, i, mkChar(theme -> name().c_str()));
		i++;
	}

	return vectorThemes;
}

void aRTdb::CreateView(string view_name)
{
	if( TeUtils::GetViewFromDatabase(Database, view_name) )
	{
		PrintSilent("There is already a view with this name\n");
		return;
	}	

	PrintSilent("Creating a view\n");
	TeProjection* proj = new TeNoProjection();
	TeView* view = new TeView(view_name, Database -> user());
	view -> projection(proj);
	PrintSilent("inserting...\n");	
	if ( !Database -> insertView(view) )
	{
		stringstream stream;
		stream << "The view \'" <<  view -> name()
			   << "\' could not be inserted into the database!\nError: "
			   << Database -> errorMessage() << "\n";
		error(StreamToChar(stream));
	}
	PrintSilent("reloading view set...\n");
//	Database -> loadViewSet( Database -> user() );
	printf("View created successfully\n");
}

SEXP aRTdb::List()
{
	SEXP vectorLayers;

	TeLayerMap& layer_map = Database -> layerMap();
	TeLayer* layer;
	TeLayerMap::iterator it;
	
	vectorLayers = layer_map.size() > 0 ? allocVector(STRSXP, layer_map.size()) : R_NilValue;
	
	int i = 0;
	for (it = layer_map.begin(); it != layer_map.end(); ++it)
	{
		layer = it -> second;
		SET_STRING_ELT(vectorLayers, i, mkChar(layer -> name().c_str()));
		i++;
    }

	return vectorLayers;
}

void aRTdb::DeleteLayer(string layer_name)
{
    TeLayer* te_layer = TeUtils::GetLayerFromDatabase(Database, layer_name);

    if (!te_layer)
        error("Layer does not exist\n");
	
	int layerid = te_layer->id();

	if( Database->deleteLayer(layerid) )
		PrintSilent("Layer removed successfully\n");
	else
		error("Layer could not be removed\n");
}

void aRTdb::DeleteView(string view_name)
{
    TeView* te_view = TeUtils::GetViewFromDatabase(Database, view_name);

    if (!te_view)
        error("View does not exist\n");

    int viewid = te_view->id();

    if( Database->deleteView(viewid) )
        PrintSilent("View removed successfully\n");
	else
    	error("View could not be removed\n");
}

void aRTdb::DeleteTheme(string theme_name)
{
	TeTheme* te_theme = TeUtils::GetThemeFromDatabase(Database, theme_name);

	if (!te_theme)
		error("Theme does not exist\n");

	int themeid = te_theme->id();

	if( Database->deleteTheme(themeid) )
		PrintSilent("Theme removed successfully\n");
	else
		error("Theme could not be removed\n");
}

aRTcomponent* aRTdb::Open(SEXP data)
{
    TeLayer* layer;
	TeProjection* proj;

	if( GetListElement(data, "themename") == RNULL )
	{
		string theme_name = GET_STRING_ELEMENT(data, "themename"); 
		return OpenTheme(theme_name);
	}
	
	string layer_name = GET_STRING_ELEMENT(data, "layername"); 
	bool create       = GET_BOOL_ELEMENT  (data, "create");
	string projection = GET_STRING_ELEMENT(data, "projection");
	
	layer = TeUtils::GetLayerFromDatabase(Database, layer_name);
	if(layer)
	{
		if (create) warning("Trying to create an existent layer\n");
		PrintSilent("Opening an existent layer\n");
	    aRTlayer* child = new aRTlayer(Database, layer_name);
	    AddChild(child);
		return child;
    }

	if(!create)
		 error("The layer does not exist! If you want to create set create=TRUE\n");
				 
	PrintSilent("Creating a new layer\n");

	proj = SEXPutils::GetTeProjection(projection);

	// also insert (in memory) the layer in Database list
   	layer = new TeLayer(layer_name, Database, proj);
    
	if (layer -> id() <= 0)
	{
		string error_ = Database -> errorMessage();
		delete proj;
		delete layer;
		error(error_.c_str());
   	}

	aRTlayer* child = new aRTlayer(Database, layer_name);
	AddChild(child);

	return child;
}

aRTcomponent* aRTdb::OpenTable(SEXP options)
{
	string table_name = GET_STRING_ELEMENT  (options, "tablename");
	bool create       = GET_BOOL_ELEMENT    (options, "create");

	aRTtable* child = new aRTexternaltable(Database, "", table_name);
	if( !Database->tableExist(table_name) )
	{
		if(!create) error("Table does not exist, set create=T to create one\n");
		child->Create(options);

		PrintSilent("Table created successfully\n");
	}
	else if(create) warning("Trying to create an existent table.\n");
	
	AddChild(child);
	return child;
}

aRTcomponent* aRTdb::OpenTheme(string theme_name)
{
	if(!TeUtils::GetThemeFromDatabase(Database, theme_name) )
		error("Theme does not exist\n");
	
	aRTtheme* child = new aRTtheme(Database, theme_name);
   	AddChild(child);
   	return child;
}

aRTcomponent* aRTdb::ImportShape(string shape_file_name, string layer_name, TeProjection* proj)
{
  	// Creates a projection, default NoProjection
  	if (proj == NULL) proj = new TeNoProjection();
  
  	TeLayer* te_layer = new TeLayer(layer_name, Database, proj);
  	aRTlayer* layer = new aRTlayer(Database, layer_name);
  	if (!te_layer || te_layer -> id() <= 0)
	{
    	layer -> SetInvalid();
    	error("Fail to create a new layer in the database");
  	}
  	else
    	AddChild(layer);
  
  	TeImportShape(te_layer, shape_file_name);
  
  	return layer;
}

enum aRTSenum
{
    aRTdbSClass = 0,
    aRTdbSDatabase,
	aRTdbSLayers,
	aRTdbSViews,
	aRTdbSThemes,
    aRTdbSSum
};

SEXP aRTdb::Summary()
{
    SEXP info, names;

    info = allocVector(VECSXP, aRTdbSSum); // List containing the connection attributes
    names = allocVector(STRSXP, aRTdbSSum); // Names Vector

    STRING_LIST_ITEM(info, names, aRTdbSClass,     "class",    Class());
	STRING_LIST_ITEM(info, names, aRTdbSDatabase,  "database", Database->databaseName());

	SEXP_LIST_ITEM(info, names, aRTdbSThemes, "themes", ShowThemes());
    SEXP_LIST_ITEM(info, names, aRTdbSViews,  "views",  ShowViews());
    SEXP_LIST_ITEM(info, names, aRTdbSLayers, "layers", List());
	
    setAttrib(info, R_NamesSymbol, names); // Set the names of the list

    return info;
}

SEXP aRTdb::ShowTables()
{
	SEXP vectorTables;
	TeAttrTableVector atts;
	TeAttributeList attrList;
	
	if ( !Database -> getAttrTables(atts, TeAttrExternal) )
		error("Error loading tables from database!");

	if( !atts.size() ) return R_NilValue;
	
	vectorTables = allocVector(STRSXP, atts.size());

	for (unsigned i = 0; i < atts.size(); i++)
		SET_STRING_ELT(vectorTables, i, mkChar(atts[i].name().c_str()));

	return vectorTables;
}

const string aRTdb::Print()
{
	stringstream s;

	if(!Database) error("Invalid object");

	TeAttrTableVector atts;
	Database->getAttrTables(atts, TeAttrExternal);

	s << artOBJECT("aRTdb") << "\n\n"
	  << "Database: \"" << Database->databaseName() << "\"" << endl
	  << "External tables: ";
    if(atts.size())
	{
		s << endl;
		for(int i = 0; i != atts.size(); i++)
			s << "    " << atts[i].name() << endl;
	}
	else s << "(none)" << endl;
			
    s << "\nThis object has " << Children.size();
	if( Children.size() != 1)	s << " children";
	else                        s << " child";
	
	s << endl;
	return s.str();
}

