/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

#include "aRT.h"
#include "SEXPbasic.h"

#include <TeUpdateDBVersion.h>

#include <string>
#include <stdlib.h>

void aRT::AddUser(string username)
{
	TeDatabase* database = NewTeDatabase();

	if( !database -> connect(Host, User, Password, "", Port))
		error("could not connect\n");

	stringstream stream;
	stream << "GRANT ALL PRIVILEGES ON *.* TO " << username << "@localhost;";
	string sql = StreamToChar(stream);
	if( !database->execute(sql) )
	{
		string err = database->errorMessage();
		delete database;
		error( err.c_str() );
	}
	
	PrintSilent("user added successfully\n");
	delete database;
}

void aRT::DeleteDb(string databasename)
{
	TeDatabase* database = NewTeDatabase();

	if( !database -> connect(Host, User, Password, "", Port))
		error("could not connect\n");
	
	stringstream stream;
	stream << "DROP DATABASE " << databasename << ";";
	string sql = StreamToChar(stream);

	if( !database->execute(sql) )
	{
		delete database;
		error( database->errorMessage().c_str() );
	}
	
	PrintSilent("database deleted successfully\n");
	delete database;
}

SEXP aRT::List()
{
	TeDatabase* database = NewTeDatabase();	
	vector<string> db_names;
	SEXP vectorDbs;

	if (!database -> showDatabases (Host, User, Password, db_names, Port))
	{
		delete database;
		return NULL;
	}

	vectorDbs = allocVector(STRSXP, db_names.size()); // Vector containing the database names

	for (unsigned i = 0; i != db_names.size(); i++)
	  SET_STRING_ELT(vectorDbs, i, mkChar(db_names[i].c_str()));

	delete database;
	return vectorDbs;
}

aRT::aRT(string user, string pass, uint port, string host, string dbms)
{
	User = user;
	Password = pass;
	Port = port;
	Host = host;
	
	if     (dbms == "mysql")   Type = aRTmySQL;
	else if(dbms == "postgre") Type = aRTpostgres;
	else error("Invalid database type: %s\n", dbms.c_str());

	TeDatabase* database = NewTeDatabase();
	vector<string> db_names;

	// the object is valid if we can list the databases
	bool Valid = database -> showDatabases(Host, User, Password, db_names, Port);
	
	if (!Valid)
	{
		string error_msg = database -> errorMessage();
		delete database;
		error(error_msg.c_str());
	}
	delete database;
}

const string aRT::Print()
{
	stringstream s;
	string ret;
	string dbms;

    if(Type == aRTmySQL)    dbms = "MySQL";
    if(Type == aRTpostgres) dbms = "PostgreSQL";
	
	s << artOBJECT("aRT")
	  << "\n\nDBMS: \""     << dbms
	  << "\"\nUser: \""     << User
	  << "\"\nPassword: \"" << Password
	  << "\"\nPort: "       << Port
	  << "\nHost: \""       << Host << "\"" << endl;
	
	return s.str();
}

// aRTSummary
enum aRTSenum
{
	aRTSClass = 0,
	aRTSUser,
	aRTSPassword,
	aRTSPort,
	aRTSHost,
	aRTSDatabases,
	aRTSSum
};

SEXP aRT::Summary()
{
	SEXP info, element, names;

	info = allocVector(VECSXP, aRTSSum); // List containing the connection attributes
	names = allocVector(STRSXP, aRTSSum); // Names Vector

    STRING_LIST_ITEM(info, names, aRTSClass,     "class",    Class());
    STRING_LIST_ITEM(info, names, aRTSUser,      "user",     User);
    STRING_LIST_ITEM(info, names, aRTSPassword,  "password", Password);
    STRING_LIST_ITEM(info, names, aRTSHost,      "host",     Host);

    INTEGER_LIST_ITEM(info, names, aRTSPort,     "port",     Port);

    SET_STRING_ELT(names, aRTSDatabases, mkChar("databases"));
    element = List();
    SET_VECTOR_ELT(info, aRTSDatabases, element);

	setAttrib(info, R_NamesSymbol, names); // Set the names of the list
	
	return info;		 
}

// Checks the database type and return a pointer to it.
TeDatabase* aRT::NewTeDatabase()
{
	switch(Type)
	{
#ifdef HAVE_MYSQL 
		case aRTmySQL:
			return new TeMySQL();
#endif
#ifdef HAVE_POSTGRE
		case aRTpostgres:
			return new TePostgreSQL();
#endif
		default:
				return NULL;
	}
	return NULL;
}

// Try to open an existent database else create a new database if (create)
aRTcomponent* aRT::Open(SEXP data)//string dbname, bool create, string str)
{
	bool silent = Silent; // **
	Silent = false; // **

    string dbname = GET_STRING_ELEMENT(data, "dbname");
	bool create   = GET_BOOL_ELEMENT(data, "create");
	bool update   = GET_BOOL_ELEMENT(data, "update");

	TeDatabase* database = NewTeDatabase();

	// ** things doesn't work if this line executes with Silent == true
	// ** strange... I already have tried to change the name of this variable.
    if (database -> connect(Host, User, Password, dbname, Port))
	{
		Silent = silent; // **
        if(create) warning("Trying to create an existent database\n");
		PrintSilent("Connecting with an existent database\n");
		string version, error_msg;
		if( needUpdateDB(database, version) )
		{
			if(update)
			{
				PrintSilent("Updating database to new terraLib version\n");
				if( updateDBVersion(database, version, error_msg) )
					PrintSilent("Database version updated successfully\n");
				else
					error("Could not update the database version: %d\n", error_msg.c_str());
			}
			else error("Database version needs to be updated. Set update=T\n");
		}
	}
	else
	{
		Silent = silent; // **
		if(!create)
			error("The database does not exist. If you want to create set create=TRUE\n");

		PrintSilent("Creating a new database\n");
        if (!database -> newDatabase(dbname, User, Password, Host, Port))
		{
	        string error_ = database -> errorMessage();
            delete database;
			error(error_.c_str());
	    }
		if (!database->createConceptualModel(true, false))
		{
			string error_ = database -> errorMessage();
			delete database;
			error(error_.c_str());
		}
	}
	
	aRTdb* root = new aRTdb(database);
	return root;
}


