/**************************************************************************\
 *  aRT : API R - TerraLib                                                *
 *  Copyright (C) 2003-2005  LEG                                          *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU Lesser General Public      *
 *  License along with this library.                                      *
\**************************************************************************/

#include "SEXPutils.h"
#include <sstream>

#ifdef USE_MYR_
	#define error(value) printf(value);
#endif

extern "C" {

namespace SEXPutils {

// visual data is a list which each attribute will be
// a data of the TeVisual
TeVisual GetTeVisual(SEXP visual, TeGeomRep geomrep)
{
	int style   = GET_INTEGER_ELEMENT(visual, "style");
	int width   = GET_INTEGER_ELEMENT(visual, "width");
	int cwidth  = GET_INTEGER_ELEMENT(visual, "cwidth");
	int transp  = GET_INTEGER_ELEMENT(visual, "transp");
	int ctransp = GET_INTEGER_ELEMENT(visual, "ctransp");
	int size    = GET_INTEGER_ELEMENT(visual, "size");
	SEXP color  = GetListElement     (visual, "color");
	SEXP ccolor = GetListElement     (visual, "ccolor");
	
	TeVisual tvisual(geomrep);

    TeColor tcolor  = SEXPToColors(color)[0];
    TeColor tccolor = SEXPToColors(ccolor)[0];
	
	tvisual.style(style);
	tvisual.size(size);
	tvisual.width(width);
	tvisual.contourWidth(cwidth);
	tvisual.transparency(transp);
	tvisual.contourTransparency(ctransp);
	tvisual.color(tcolor); 
	tvisual.contourColor(tccolor);
	
	return tvisual;
}

// um unico espao separando os atributos e sem \n
TeProjection* GetTeProjection(string proj4)
{
	return TeGetTeProjectionFromSProj(proj4);
}
		
vector<TeColor> SEXPToColors(SEXP colors)
{
    vector<TeColor> colorVec;
    int size = LENGTH(colors)/3;
    
    for(int i = 0; i < size; i++)
    {
        TeColor cor(INTEGER(colors)[3*i],
                    INTEGER(colors)[3*i+1], 
                    INTEGER(colors)[3*i+2]);
        colorVec.push_back(cor);
    }
    return colorVec;
}

void SetColors(TeRasterParams& params, vector<TeColor> colors)
{
    params.setNumberPalleteEntries(colors.size());
    
    for(unsigned i = 0; i < colors.size(); i++)
    {
        params.lutr_[i] = colors[i].red_;
        params.lutg_[i] = colors[i].green_;
        params.lutb_[i] = colors[i].blue_;
    }
}

bool AddRaster(TeLayer* layer, SEXP rasterdata)
{
	TeInitRasterDecoders();
    TeRasterParams params;

    SEXP x   = GetListElement(rasterdata, "x");
	SEXP y   = GetListElement(rasterdata, "y");
    SEXP dim = GetListElement(rasterdata, "dim");
	
	TeProjection* proj = new TeNoProjection();

    params.nBands(1);
    params.projection(proj);
    params.nlines_ = INTEGER(dim)[1];
    params.ncols_ = INTEGER(dim)[0];

	params.boundingBoxLinesColumns(REAL(x)[0],
				        		   REAL(y)[0],
								   REAL(x)[1],
								   REAL(y)[1],
								   INTEGER(dim)[1],
								   INTEGER(dim)[0]);
    params.setDataType(TeDOUBLE);
    params.setPhotometric(TeRASTERKERNEL);
    params.decoderIdentifier_ = "MEM";
    params.mode_ = 'c';

	TeDecoderSEXP* decoder = new TeDecoderSEXP(rasterdata, params);
	TeRaster* raster = new TeRaster();
    raster->setDecoder(decoder);

    if(!raster->init()) return false;
    return TeImportRaster (layer,
                           raster,
                           INTEGER(dim)[0],
                           INTEGER(dim)[1],
                           TeNoCompression,
                           "", // object id
						   raster->params().dummy_[0],
						   true); // usedummy
}

SEXP GetBoxTeBox(TeBox box){
  SEXP *coords, rownames, colnames, dimnames;
	SEXP result;

    PROTECT(result   = allocVector(VECSXP, 2));
    PROTECT(colnames = allocVector(STRSXP, 2));

    coords = new SEXP[2];

    PROTECT(coords[0] = allocVector(REALSXP,5));
    PROTECT(coords[1] = allocVector(REALSXP,5));

    SET_STRING_ELT( colnames, 0, mkChar("x") );
    SET_STRING_ELT( colnames, 1, mkChar("y") );

    REAL(coords[0])[0] = box.x1();
    REAL(coords[1])[0] = box.y1();

    REAL(coords[0])[1] = box.x2();
    REAL(coords[1])[1] = box.y1();

    REAL(coords[0])[2] = box.x2();
    REAL(coords[1])[2] = box.y2();

    REAL(coords[0])[3] = box.x1();
    REAL(coords[1])[3] = box.y2();

	REAL(coords[0])[4] = box.x1();
    REAL(coords[1])[4] = box.y1();

    SET_VECTOR_ELT(result, 0, coords[0]);
    SET_VECTOR_ELT(result, 1, coords[1]);

    setAttrib(result, R_NamesSymbol, colnames);
    delete[] coords;

    UNPROTECT(4);

    return result;
}              

SEXP GetPointsUtils(TePointSet& ps)
{
  int nroptos = ps.size() ;
  SEXP coords, rownames, colnames, dimnames;

  PROTECT(coords = allocMatrix(REALSXP, 2, nroptos));
  PROTECT(rownames = allocVector(INTSXP, nroptos));
  PROTECT(dimnames = allocVector(VECSXP, 2));
  PROTECT(colnames = allocVector(STRSXP, 2));

  SET_STRING_ELT(colnames, 0, mkChar("x"));
  SET_STRING_ELT(colnames, 1, mkChar("y"));

  for (int j = 0; j < nroptos; j++) {
    REAL(coords)[0 + 2 * j] = ps[j].location().x();
    REAL(coords)[1 + 2 * j] = ps[j].location().y();

    const char *id = ps[j].objectId().c_str();

    // creates a vector with values rownames
    INTEGER(rownames)[j] = atoi(id);
  }
    
  SET_VECTOR_ELT(dimnames, 0, colnames);
  SET_VECTOR_ELT(dimnames, 1, rownames);
  setAttrib(coords, R_DimNamesSymbol, dimnames);
  UNPROTECT(4);
  return coords;
}

SEXP GetLayerData(TeLayer* layer)
{
	return R_NilValue;
}

SEXP GetThemeData(TeTheme* theme)
{
	return R_NilValue;
}

SEXP GetLayerAttributes(TeLayer* layer)
{
    TeInitQuerierStrategies(); // never forget it!

    TeQuerierParams params; // params default is no geometry, only attributes
    params.setParams(layer);
    TeQuerier* querier = new TeQuerier(params);
	SEXP result = GetQuerierAttributes(querier);
	delete querier;
	return result;
}

SEXP GetThemeAttributes(TeTheme* theme)
{
    TeInitQuerierStrategies(); // never forget it!

    TeQuerierParams params; // params default is no geometry, only attributes
    params.setParams(theme);
    TeQuerier* querier = new TeQuerier(params);
	SEXP result = GetQuerierAttributes(querier);
	delete querier;
	return result;
}

SEXP GetQuerierAttributes(TeQuerier* querier)
{
	SEXP colnames, result;
	SEXP* each_column;
	TeSTInstance sti;
	TePropertyVector pv;
	int size = 0;
	int lines = 0;
	
	if( !querier->loadInstances() )    return R_NilValue;
	if( !querier->fetchInstance(sti) ) return R_NilValue;

	// first time we get an instance and set the colnames...
	size = sti.getPropertyVector().size();
	pv = sti.getPropertyVector();
	
	const int instances =  querier->numElemInstances();
	colnames    = PROTECT(allocVector(VECSXP, size));
	result      = PROTECT(allocVector(VECSXP, size));
	each_column = new SEXP[size];

	for(int i = 0; i < size; i++)
	{
		switch(pv[i].attr_.rep_.type_)
		{
			case TeREAL:
				each_column[i] = PROTECT(allocVector(REALSXP,instances));
				break;
			case TeINT:
				each_column[i] = PROTECT(allocVector(INTSXP,instances));
				break;
			default:
				each_column[i] = PROTECT(allocVector(STRSXP,instances));
				break;
		}
		SET_STRING_ELT( colnames, i, mkChar(pv[i].attr_.rep_.name_.c_str()) );
	}
	
	// ... then we set all the values.
	// The while is at the end because we already read the first value
	do
	{
		pv = sti.getPropertyVector();
		for(int i = 0; i < size; i++)
		{
			// check the type, not all times will be real
	        switch(pv[i].attr_.rep_.type_)
    	    {
        	    case TeREAL:
       				REAL(each_column[i])[lines] = atof(pv[i].value_.c_str());
					break;
	            case TeINT:
       				INTEGER(each_column[i])[lines] = atoi(pv[i].value_.c_str());
					break;
        	    default:
       				SET_STRING_ELT(each_column[i], lines, mkChar(pv[i].value_.c_str()));
					break;
	        }
		}
		sti.clear();
		lines++;
	} while( querier->fetchInstance(sti) );
	
	setAttrib(result, R_NamesSymbol, colnames);

    for(int i = 0; i < size; i++) SET_VECTOR_ELT(result, i, each_column[i]);
	
	UNPROTECT(2 + size); // data and colnames, and onde more per attribute
	
	delete[] each_column;
	return result;
}

SEXP GetLinesTeMultiGeometry(TeMultiGeometry& mg)
{
	TeLineSet lineset;
	mg.getGeometry(lineset);

	return GetLinesTeLineSet(lineset);
}

SEXP GetLinesTeLineSet(TeLineSet& lineset)
{
	SEXP result, colnames;
	SEXP* each_column;
	int count = 0, size = 0;

    // count the number of points in each polygon
    for(unsigned i = 0; i < lineset.size(); i++)
        size+=lineset[i].size();

	PROTECT(result   = allocVector(VECSXP, 2));
	PROTECT(colnames = allocVector(STRSXP, 2));
	
	each_column = new SEXP[2];

	PROTECT(each_column[0] = allocVector(REALSXP,size));
	PROTECT(each_column[1] = allocVector(REALSXP,size));
	SET_STRING_ELT( colnames, 0, mkChar("x") );
	SET_STRING_ELT( colnames, 1, mkChar("y") );
	
	// set the values in result
	for(unsigned i = 0; i < lineset.size(); i++)
	{
        for(unsigned j = 0; j < lineset[i].size(); j++)
		{
			REAL(each_column[0])[count] = lineset[i][j].x();
			REAL(each_column[1])[count] = lineset[i][j].y();
			count++;
		}
	}
	SET_VECTOR_ELT(result, 0, each_column[0]);	
	SET_VECTOR_ELT(result, 1, each_column[1]);	

	setAttrib(result, R_NamesSymbol, colnames);
	delete[] each_column;

	UNPROTECT(4);

	return result;
}

SEXP GetPolygonsTeMultiGeometry(TeMultiGeometry& mg)
{
	TePolygonSet polset;
	mg.getGeometry(polset);
    
	return GetPolygonsTePolygonSet(polset);
}

SEXP GetPolygonsTePolygonSet(TePolygonSet& polset)
{
	SEXP result, colnames;
	SEXP* each_column;
	int count = 0, size = 0;
		
	// count the number of points in each polygon
	for(unsigned i = 0; i < polset.size(); i++)
		for(unsigned j = 0; j < polset[i].size(); j++)
			size+=polset[i][j].size();

	PROTECT(result   = allocVector(VECSXP, 2));
	PROTECT(colnames = allocVector(STRSXP, 2));
	
	each_column = new SEXP[2];

	PROTECT(each_column[0] = allocVector(REALSXP,size));
	PROTECT(each_column[1] = allocVector(REALSXP,size));
	SET_STRING_ELT( colnames, 0, mkChar("x") );
	SET_STRING_ELT( colnames, 1, mkChar("y") );
	
	// set the values in result
	for(unsigned i = 0; i < polset.size(); i++)
	{
        for(unsigned j = 0; j < polset[i].size(); j++)
            for(unsigned k = 0; k < polset[i][j].size(); k++)
			{
				REAL(each_column[0])[count] = polset[i][j][k].x();
				REAL(each_column[1])[count] = polset[i][j][k].y();
				count++;
			}
	}
	SET_VECTOR_ELT(result, 0, each_column[0]);	
	SET_VECTOR_ELT(result, 1, each_column[1]);	

	setAttrib(result, R_NamesSymbol, colnames);
	delete[] each_column;

	UNPROTECT(4);

	return result;
}

SEXP GetPointsTeMultiGeometry(TeMultiGeometry& mg)
{
	TePointSet ps;
	mg.getGeometry(ps);

    return GetPointsTePointSet(ps);
}

SEXP GetPointsTePointSet(TePointSet& ps)
{
	const int size = ps.size() ;
	SEXP result, colnames, *each_column;
	
	PROTECT(result   = allocVector(VECSXP, 2));
	PROTECT(colnames = allocVector(STRSXP, 2));
	
	each_column = new SEXP[2];

	PROTECT(each_column[0] = allocVector(REALSXP,size));
    PROTECT(each_column[1] = allocVector(REALSXP,size));
	SET_STRING_ELT(colnames, 0, mkChar("x"));
	SET_STRING_ELT(colnames, 1, mkChar("y"));

	for (int j = 0; j < size; j++)
	{
        REAL(each_column[0])[j] = ps[j].location().x();
        REAL(each_column[1])[j] = ps[j].location().y();
    }
    SET_VECTOR_ELT(result, 0, each_column[0]);
    SET_VECTOR_ELT(result, 1, each_column[1]);

    setAttrib(result, R_NamesSymbol, colnames);
    delete[] each_column;

	UNPROTECT(4);
	return result;
}

TePointSet GetTePointSet(SEXP sps)
{
	TePointSet ps;

	SEXP ids  = GetListElement(sps, aRTstrId);
	SEXP geom = GetListElement(sps, aRTstrPoint);
	
	const int size = LENGTH(ids);

	for(int i = 0; i < size; i++)
	{
		if(VECTOR_ELT(geom,i) != R_NilValue)
		{
			SEXP arx = GetListElement(VECTOR_ELT(geom,i), "x");
			SEXP ary = GetListElement(VECTOR_ELT(geom,i), "y");

			const int internal_size = LENGTH(arx);

			for(int j = 0; j < internal_size; j++) // one id can have more than one point
			{
				TePoint point(REAL(arx)[j], REAL(ary)[j]);
				point.objectId( CHAR(STRING_ELT(ids, i)) );
				ps.add(point);
			}
		}
	}
	return ps;
}

TePolygonSet GetTePolygonSet(SEXP sps)
{
    TePolygonSet ps;

	SEXP ids  = GetListElement(sps, aRTstrId);
    SEXP geom = GetListElement(sps, aRTstrPolygon);

    const int size = LENGTH(ids);

    for(int i = 0; i < size; i++)
    {
		if(VECTOR_ELT(geom,i) != R_NilValue)
		{
        	SEXP arx = GetListElement(VECTOR_ELT(geom,i), "x");
        	SEXP ary = GetListElement(VECTOR_ELT(geom,i), "y");

        	const int internal_size = LENGTH(arx);

			for(int j = 0; j < internal_size; )
			{
				TeLine2D line;

    	        TeCoord2D coord(REAL(arx)[j], REAL(ary)[j]);
				line.add(coord);
				j++;
	
				do
				{
	            	TeCoord2D coord(REAL(arx)[j], REAL(ary)[j]);
					line.add(coord);
					j++;
				} while (line.last() != line.first());
			
				TeLinearRing lr(line);
				TePolygon pol;
				pol.objectId( CHAR(STRING_ELT(ids, i)) );
	
				pol.add(lr);
				ps.add(pol);
				//line.clear(); // <= ESSA MERDA!
	        }
		}
    }
	return ps;
}

TeLineSet GetTeLineSet(SEXP sls)
{
    TeLineSet ls;
	SEXP ids  = GetListElement(sls, aRTstrId);
    SEXP geom = GetListElement(sls, aRTstrLine);

    const int size = LENGTH(ids);

    for(int i = 0; i < size; i++)
    {
		if(VECTOR_ELT(geom,i) != R_NilValue)
		{
        	SEXP arx = GetListElement(VECTOR_ELT(geom,i), "x");
        	SEXP ary = GetListElement(VECTOR_ELT(geom,i), "y");

        	const int internal_size = LENGTH(arx);
			TeLine2D line;

			// TODO: more than one line per object
			for(int j = 0; j < internal_size; j++)
			{
        	    TeCoord2D coord(REAL(arx)[j], REAL(ary)[j]);
				line.add(coord);
			}

			line.objectId( CHAR(STRING_ELT(ids, i)) );
			ls.add(line);
		}
    }
	return ls;
}

SEXP GetCellsTeMultiGeometry(TeMultiGeometry&)
{
	return R_NilValue;
}

enum Geometries
{
	GeomPoints,
	GeomPolygons,
	GeomLines,
	GeomSize
};

const char* GeometryNames[] =
{
    aRTstrPoint,
    aRTstrPolygon,
	aRTstrLine
};

SEXP GetLayerGeometry(TeLayer* layer)
{
    TeInitQuerierStrategies(); // never forget it!
    
    TeQuerierParams params(true, false); // only geometry, no attributes
    params.setParams(layer);
    TeQuerier* querier = new TeQuerier(params);
	
	SEXP result = GetQuerierGeometry(querier);
	delete querier;
	return result;
}

SEXP GetThemeGeometry(TeTheme* theme)
{
    TeInitQuerierStrategies(); // never forget it!
    
    TeQuerierParams params(true, false); // only geometry, no attributes
    params.setParams(theme);
    TeQuerier* querier = new TeQuerier(params);
	
	SEXP result = GetQuerierGeometry(querier);
	delete querier;
	return result;
}

SEXP GetQuerierGeometry(TeQuerier* querier)
{	
	SEXP colnames, result;
	TeSTInstance sti;
	TeMultiGeometry mg;
	int lines = 0;
	int geometries = 0;
	SEXP vgeometries[GeomSize];
	SEXP ids;
	
	for(int i = 0; i < GeomSize; i++)
		vgeometries[i] = R_NilValue;
	
	if( !querier->loadInstances() )
	{
		cout << "can't load instances" << endl;
		return R_NilValue;
	}
	if( !querier->fetchInstance(sti) )
	{
		cout << "can't fetch instances" << endl;
		return R_NilValue;
	}

	const int instances =  querier->numElemInstances();

	#define IF_NIL_ALLOC_AND_SET(object)           \
	if(object == R_NilValue)                       \
	{                                              \
		object = allocVector(VECSXP, instances);   \
		PROTECT(object);                           \
	    geometries ++;                             \
		for(int i = 0; i < instances; i++)         \
	        SET_VECTOR_ELT(object, i, R_NilValue); \
	}
	
    PROTECT(ids = allocVector(STRSXP, instances));

	do{
		mg.clear();
		mg = sti.geometries();
   
		SET_STRING_ELT(ids, lines, mkChar(sti.objectId().c_str()));

		if( mg.hasPolygons() )
		{   // an instance can have more then one polygon. In this case,
			// each one is connected with the previous and the next by
			// "invisible" edges (except the first and the last, that have only one)
			// the first point of a polygon is repeated at its end
			IF_NIL_ALLOC_AND_SET(vgeometries[GeomPolygons]);
			SET_VECTOR_ELT(vgeometries[GeomPolygons], lines, GetPolygonsTeMultiGeometry(mg));
		}
		
		if( mg.hasPoints() )
		{
			IF_NIL_ALLOC_AND_SET(vgeometries[GeomPoints]);
			SET_VECTOR_ELT(vgeometries[GeomPoints], lines, GetPointsTeMultiGeometry(mg));
		}

		if( mg.hasLines() )
		{
			IF_NIL_ALLOC_AND_SET(vgeometries[GeomLines]);
			SET_VECTOR_ELT(vgeometries[GeomLines], lines, GetLinesTeMultiGeometry(mg));
		}
		if( mg.hasCells() ) cout << "cells not supported yet" << endl;
		if( mg.empty() )    cout << "found an empty geometry: "  << lines << endl;
		mg.clear();
		sti.clear();
		lines++;
	} while( querier->fetchInstance(sti) );

	if(geometries)
	{
	    PROTECT(colnames = allocVector(STRSXP, geometries+1));
	    PROTECT(result   = allocVector(VECSXP, geometries+1));
		
		int j = 0;
		for(int i = 0; i != GeomSize; i++)
		{
			if(vgeometries[i] != R_NilValue)
			{
			    SET_STRING_ELT(colnames, j, mkChar(GeometryNames[i]));
				SET_VECTOR_ELT(result,   j, vgeometries[i]);
				j++;
			}
		}

		SET_STRING_ELT(colnames, j, mkChar(aRTstrId));
		SET_VECTOR_ELT(result,   j, ids);

		setAttrib(result, R_NamesSymbol, colnames);

		UNPROTECT(geometries + 3); // + colnames + result + ids
		return result;
	}
	cout << "get geom erro" << endl;
	UNPROTECT(geometries + 1);
	return R_NilValue;
}

void printAttrib(SEXP e)
{
	SEXP rn=Rf_GetRowNames(e);
	if(rn != R_NilValue)
	{
		printf("Row names:\n");
		printSEXP(rn);
	}

	SEXP cn=getAttrib(e, R_NamesSymbol);
	{
		printf("Col names:\n");
		printSEXP(cn);
	}
}

void printSEXP(SEXP e)
{
    int i = 0;

    switch( TYPEOF(e) )
    {
        case NILSXP:
            printf("NULL value\n");
            return;

        case LANGSXP:
            printf("language construct\n");
            return;

        case REALSXP:
            if (LENGTH(e) > 1)
            {
                printf("Vector of real variables: ");
                while( i < LENGTH(e) )
                {
                    printf("%f",REAL(e)[i]);
                    if ( !(REAL(e)[i] < 0) && !(REAL(e)[i] >= 0) ) printf("__NAN__");
                    if (i < LENGTH(e) - 1) printf(", ");
                    if (dumpLimit && i > dumpLimit) { printf("..."); break; }
                    i++;
                }
                putchar('\n');
            }
            else printf( "Real variable %f\n", *REAL(e) );
            return;
        case EXPRSXP:
            printf( "Vector of %d expressions:\n", LENGTH(e) );
            while( i < LENGTH(e) )
            {
                if (dumpLimit && i > dumpLimit){ printf("...");  break; }
                printSEXP( VECTOR_ELT(e,i) );
                i++;
            }
            return;

         case INTSXP:
            printf( "Vector of %d integers:\n", LENGTH(e) );

            while( i < LENGTH(e) )
            {
                if (dumpLimit && i > dumpLimit) { printf("..."); break; }
                printf("%d",INTEGER(e)[i]);
                if (i < LENGTH(e) - 1) printf(", ");
                i++;
            };
            putchar('\n');
            return;

        case VECSXP:
            printf( "Vector of %d fields:\n", LENGTH(e) );
            while( i < LENGTH(e) )
            {
                if (dumpLimit && i>dumpLimit) { printf("..."); break; };
                printSEXP(VECTOR_ELT(e,i));
                i++;
            }
            return;

        case STRSXP:
            printf( "String vector of length %d:\n", LENGTH(e) );
            while( i < LENGTH(e) )
            {
                if (dumpLimit && i > dumpLimit) { printf("..."); break; };
                printSEXP(VECTOR_ELT(e,i)); i++;
            }
            return;
	   case CHARSXP:
            printf( "scalar string: \"%s\"\n",(char*) STRING_PTR(e) );
            return;

        case SYMSXP:
            printf( "Symbol, name: "); printSEXP(PRINTNAME(e) );
            return;

        default:
            printf( "Unknown type: %d\n", TYPEOF(e) );
    }
};

string AttrDataType(TeAttrDataType value)
{
    switch (value)
    {
        case TeSTRING:    return "string";
        case TeREAL:      return "real";
        case TeINT:       return "integer";
        case TeDATETIME:  return "date";
        case TeBLOB:      return "blob";
        case TeOBJECT:    return "object";
        case TeCHARACTER: return "character";
        case TeUNKNOWN:   return "<unknown>";
    }
}

string AttrTableType(TeAttrTableType value)
{
    switch(value)
    {
        case TeAttrStatic:       return "static";
        case TeAttrMedia:        return "media";
        case TeAttrExternal:     return "external";
        case TeAttrEvent:        return "event";
        case TeFixedGeomDynAttr: return "dynattr";
        case TeDynGeomDynAttr:   return "alldyn";
        case TeGeomAttrLinkTime: return "dyngeom";
        case TeGeocodingData:
        default:                 return "<unrecognized type>";
    }
}

}

}

