positive.poisson <- local({

  lambda <- local({
    lam1 <- function(mu, eps) {
              lam <- mu*(1 - exp(1/mu - mu))
              while (any(abs(del <- (lam - mu * (1 - exp(-lam)))/
                          (1 - mu * exp(-lam))) > eps))
                lam <- lam - del
              lam - del
            }
    function(mu, eps = sqrt(.Machine$double.eps)) {
      lam <- numeric(length(mu))
      if(any(k <- mu < 1)) {
        is.na(lam[k]) <- TRUE
        warning("missing values generated")
      }
      lam[mu == 1] <- 0
      if(any(k <- mu > 1)) lam[k] <- lam1(mu[k], eps)
      lam
    }
  })

  makeLink <- function(link) {
    switch(link,
      canonical = {
        linkfun <- function(mu) log(lambda(mu))
        linkinv <- function(eta) exp(eta)/(1 - exp(-exp(eta)))
        mu.eta <- function(eta) pmax(.Machine$double.eps,
            exp(eta)/(1 - exp(-exp(eta))) -
            exp(2 * eta - exp(eta))/(1 - exp(-exp(eta)))^2)
        valideta <- function(eta) TRUE
      },
      "log(mu-1)" = {
        linkfun <- function(mu) log(mu-1)
        linkinv <- function(eta) 1 + exp(eta)
        mu.eta <- function(eta) pmax(.Machine$double.eps, exp(eta))
        valideta <- function(eta) TRUE
      }, identity = {
        linkfun <- function(mu) mu
        linkinv <- function(eta) eta
        mu.eta <- function(eta) rep.int(1, length(eta))
        valideta <- function(eta) all(eta >= 0)
      }, log = {
        linkfun <- function(mu) log(mu)
        linkinv <- function(eta) pmax(exp(eta), .Machine$double.eps)
        mu.eta <- function(eta) pmax(exp(eta), .Machine$double.eps)
        valideta <- function(eta) TRUE
      }, sqrt = {
        linkfun <- function(mu) sqrt(mu)
        linkinv <- function(eta) eta^2
        mu.eta <- function(eta) 2 * eta
        valideta <- function(eta) all(eta > 0)
      }, "sqrt(mu-1)" = {
        linkfun = function(mu) sqrt(mu-1)
        linkinv <- function(eta) 1 + eta^2
        mu.eta <- function(eta) 2 * eta
        valideta <- function(eta) all(eta > 0)
      })
      structure(list(linkfun = linkfun,
                     linkinv = linkinv,
                     mu.eta = mu.eta,
                     valideta = valideta,
                     name = link), class = "link-glm")
  }
  links <- c("canonical", "log(mu-1)", "identity", "log", "sqrt", "sqrt(mu-1)")
  
  function(link = "canonical") {
    linktemp <- substitute(link)
    if(!is.character(linktemp)) linktemp <- deparse(linktemp)
    linktemp <- links[pmatch(linktemp, links)]
    if(is.na(linktemp))
      stop("No '", linktemp, "' link is available for the positive poisson family")
    Link <- makeLink(linktemp)
    variance <- function(mu) mu*(1 + lambda(mu) - mu)
    validmu <- function(mu) all(mu > 1)
    dev.resids <- function(y, mu, wt) {
      ly <- lambda(y)
      lu <- lambda(mu)
      2 * wt * ((y-1) * log((ly + (y == 1))/lu) + log(y/mu) - (ly - lu))
    }
    aic <- function(y, n, mu, wt, dev) {
      lu <- lambda(mu)
      term <- (y - 1)*log(lu) + log(mu) - lu - lgamma(y+1)
      -2 * sum(term * wt)
    }
    initialize <- expression({
      if(any(y < 1)) stop("observations less than 1 are not allowed")
      n <- rep(1, nobs)
      mustart <- y + (y == 1)/6
    })
    famname <- "positive.poisson"
    with(Link, structure(list(family = famname,
                          link = linktemp,
                          linkfun = linkfun,
                          linkinv = linkinv,
                          variance = variance,
                          dev.resids = dev.resids,
                          aic = aic,
                          mu.eta = mu.eta,
                          initialize = initialize,
                          validmu = validmu,
                          valideta = valideta),
                class = "family")
    )
  }
})