 
#include "TePDINoBlendStrategy.hpp"
#include "TePDIBlending.hpp"
#include "TePDIUtils.hpp"
#include "TePDIStrategyFactory.hpp"
#include "TePDIPIManager.hpp"
#include <TeAgnostic.h>

#include <TeBox.h>
#include <TeUtils.h>
#include <TeOverlay.h>

TePDINoBlendStrategy::TePDINoBlendStrategy()
{
};      

TePDINoBlendStrategy::~TePDINoBlendStrategy()
{
};


bool TePDINoBlendStrategy::CheckParameters( 
  const TePDIParameters& parameters ) const
{
  /* Checking input_raster1 */
  
  TePDITypes::TePDIRasterPtrType input_raster1;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster1", 
    input_raster1 ),
    "Missing parameter: input_raster1" );
  TEAGN_TRUE_OR_RETURN( input_raster1.isActive(),
    "Invalid parameter: input_raster1 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster1->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster1 not ready" );    
    
  /* Checking input_raster2 */
  
  TePDITypes::TePDIRasterPtrType input_raster2;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster2", 
    input_raster2 ),
    "Missing parameter: input_raster2" );
  TEAGN_TRUE_OR_RETURN( input_raster2.isActive(),
    "Invalid parameter: input_raster2 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster2->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster2 not ready" );    
    
  /* Checking output_raster */
  
  TePDITypes::TePDIRasterPtrType output_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "output_raster", 
    output_raster ),
    "Missing parameter: output_raster" );
  TEAGN_TRUE_OR_RETURN( output_raster.isActive(),
    "Invalid parameter: output_raster inactive" );
  TEAGN_TRUE_OR_RETURN( output_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: output_raster not ready" );    
    
  /* channels1 parameter checking */

  std::vector< int > channels1;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels1", channels1 ), 
    "Missing parameter: channels1" );
  for( unsigned int channels1_index = 0 ; 
    channels1_index < channels1.size() ; 
    ++channels1_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels1[ channels1_index ] >= 0 ) &&
      ( channels1[ channels1_index ] < input_raster1->nBands() ),
      "Invalid parameter: channels1" );
  }
  
  /* channels2 parameter checking */

  std::vector< int > channels2;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels2", channels2 ), 
    "Missing parameter: channels2" );
  TEAGN_TRUE_OR_RETURN( ( channels2.size() == channels1.size() ),
    "Size mismatch between channels1 and channels2" );
  for( unsigned int channels2_index = 0 ; 
    channels2_index < channels2.size() ; 
    ++channels2_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels2[ channels2_index ] >= 0 ) &&
      ( channels2[ channels2_index ] < input_raster2->nBands() ),
      "Invalid parameter: channels2" );
  }       
    
  /* Checking raster polygons */
  
  TePDITypes::TePDIPolygonPtrType raster1_pol_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster1_pol_ptr", 
    raster1_pol_ptr ), "Missing parameter : raster1_pol_ptr" );
  TEAGN_TRUE_OR_RETURN( raster1_pol_ptr.isActive(),
    "Invalid parameter : raster1_pol_ptr" )
  TEAGN_TRUE_OR_RETURN(
    ( ! input_raster1->begin( *raster1_pol_ptr, TeBoxPixelIn, 
    0 ).end() ), "Invalid parameter : raster1_pol_ptr" )
  TEAGN_TRUE_OR_RETURN( ( raster1_pol_ptr->size() == 1 ),
    "Invalid parameter : raster1_pol_ptr" )
    
  TePDITypes::TePDIPolygonPtrType raster2_pol_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_ptr", 
    raster2_pol_ptr ), "Missing parameter : raster2_pol_ptr" );
  TEAGN_TRUE_OR_RETURN( raster2_pol_ptr.isActive(),
    "Invalid parameter : raster2_pol_ptr" )
  TEAGN_TRUE_OR_RETURN(
    ( ! input_raster2->begin( *raster2_pol_ptr, TeBoxPixelIn, 
    0 ).end() ), "Invalid parameter : raster2_pol_ptr" )
  TEAGN_TRUE_OR_RETURN( ( raster2_pol_ptr->size() == 1 ),
    "Invalid parameter : raster2_pol_ptr" )
    
  /* Checking raster2 polygon offsets */
  
  double raster2_pol_offset_x = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_offset_x", 
    raster2_pol_offset_x ), "Missing parameter : raster2_pol_offset_x" );  

  double raster2_pol_offset_y = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_offset_y", 
    raster2_pol_offset_y ), "Missing parameter : raster2_pol_offset_y" );    
    
  return true;
}


bool TePDINoBlendStrategy::Implementation( const TePDIParameters& params )
{
  /* Extracting parameters */

  TePDITypes::TePDIRasterPtrType input_raster1;
  params.GetParameter( "input_raster1", input_raster1 );
  TeRaster& input_raster1_ref = *( input_raster1.nakedPointer() );

  TePDITypes::TePDIRasterPtrType input_raster2;
  params.GetParameter( "input_raster2", input_raster2 );
  TeRaster& input_raster2_ref = *( input_raster2.nakedPointer() );

  TePDITypes::TePDIRasterPtrType output_raster;
  params.GetParameter( "output_raster", output_raster );
  TeRaster& output_raster_ref = *( output_raster.nakedPointer() );
  
  std::vector< int > channels1;
  params.GetParameter( "channels1", channels1 );

  std::vector< int > channels2;
  params.GetParameter( "channels2", channels2 ); 
 
  /* Dummy definition */
  
  const bool no_force_dummy = ! params.CheckParameter< double >( "dummy_value" );
  double forced_dummy_value = 0;
  if( ! no_force_dummy ) {
    params.GetParameter( "dummy_value", forced_dummy_value );
  }    
  
  /* Calculating intersecion polygons data */
  
  TePolygon new_raster1_polygon;
  TePolygon new_raster2_polygon;
  TePolygon int_pol_refout;
  TePolygon int_pol_ref2;
  double raster1_rel_index_offset_x = 0;
  double raster1_rel_index_offset_y = 0;
  double raster2_rel_index_offset_x = 0;
  double raster2_rel_index_offset_y = 0;
  short pols_relation = 0;

  TEAGN_TRUE_OR_RETURN( TePDIBlending::extractPolygons( params, 
    new_raster1_polygon, new_raster2_polygon, int_pol_refout, int_pol_ref2,
    pols_relation,
    raster1_rel_index_offset_x, raster1_rel_index_offset_y,
    raster2_rel_index_offset_x, raster2_rel_index_offset_y ),
    "Error extracting intersection polygons" );
  
  /* Global vars */
  
  TeRaster::iteratorPoly intersection_refout_it =
    output_raster->begin( int_pol_refout, TeBoxPixelIn, 0 );
  TEAGN_TRUE_OR_RETURN( ( ! intersection_refout_it.end() ), 
    "Unable to create an iterator over raster2 area" );       
  
  TePDIPIManager progress( "Rendering intersection...", 
    intersection_refout_it.nLinesInPoly() * channels1.size(),
    progress_interface_enabled_ );
    
  /* Rendering intersection */
  
  int raster1_offset_x = TeRound( raster1_rel_index_offset_x );
  int raster1_offset_y = TeRound( raster1_rel_index_offset_y );    
  int raster2_offset_x = TeRound( raster2_rel_index_offset_x );
  int raster2_offset_y = TeRound( raster2_rel_index_offset_y );    
  
  for( unsigned int channels_index = 0 ; channels_index < channels1.size() ;
    ++channels_index ) {
    
    const unsigned int channel1 = channels1[ channels_index ];
    const unsigned int channel2 = channels2[ channels_index ];  
  
    unsigned int curr_line = 0;
    unsigned int curr_col = 0;
    unsigned int last_line = 0; 
    
    double value = 0;
    
    intersection_refout_it =
      output_raster->begin( int_pol_refout, TeBoxPixelIn, 0 );
  
    while( ! intersection_refout_it.end() ) {
      curr_line = intersection_refout_it.currentLine();
      curr_col = intersection_refout_it.currentColumn();
            
      if( input_raster2_ref.getElement( curr_col + raster2_offset_x, 
        curr_line + raster2_offset_y, value, channel2 ) ) {
        
        if( no_force_dummy || ( value != forced_dummy_value ) ) {
          TEAGN_TRUE_OR_RETURN( 
              output_raster_ref.setElement( curr_col, curr_line, value, 
              channels_index ),
              "Unable to write to output raster at line=" + 
              Te2String( curr_line ) + 
              " col=" + Te2String( curr_col ) );          
        } else {
          if( input_raster1_ref.getElement( curr_col + raster1_offset_x, 
            curr_line + raster1_offset_y, value, channel1 ) ) {
            
            if( no_force_dummy || ( value != forced_dummy_value ) ) {
              TEAGN_TRUE_OR_RETURN( 
                output_raster_ref.setElement( curr_col, curr_line, value, 
                channels_index ),
                "Unable to write to output raster at line=" + 
                Te2String( curr_line ) + 
                " col=" + Te2String( curr_col ) );            
            }
          }     
        }
      } else {
        if( input_raster1_ref.getElement( curr_col + raster1_offset_x, 
          curr_line + raster1_offset_y, value, channel1 ) ) {
          
          if( no_force_dummy || ( value != forced_dummy_value ) ) {
            TEAGN_TRUE_OR_RETURN( 
              output_raster_ref.setElement( curr_col, curr_line, value, 
              channels_index ),
              "Unable to write to output raster at line=" + 
              Te2String( curr_line ) + 
              " col=" + Te2String( curr_col ) );            
          }
        }
      }
      
      if( curr_line != last_line ) {
        last_line = curr_line;
        progress.Increment();
      }      
    
      ++intersection_refout_it;
    }
  }
  
  return true;
}


