/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#include "TeGeometricTransformation.h"
#include "TeMutex.h"
#include "TeThreadFunctor.h"
#include "TeUtils.h"
#include "TeAgnostic.h"

#include <math.h>
#include <float.h>

#include <map>
#include <algorithm>


TeGeometricTransformation::TeGeometricTransformation()
{
}


TeGeometricTransformation::~TeGeometricTransformation()
{
}


void TeGeometricTransformation::getParameters( TeGTParams& params ) const
{
  params = internal_params_;
}


bool TeGeometricTransformation::reset( const TeGTParams& newparams )
{
  /* If previous calculated parameters were supplied, no need to do calcules */

  if( isTransDefined( newparams ) ) 
  {
    internal_params_ = newparams;
    return true;
  } else {
    /* No previous parameters given - Need to calculate the new transformation
       parameters */
       
    TEAGN_TRUE_OR_THROW( ( newparams.max_dmap_error_ >= 0 ),
      "Invalid maximum allowed direct mapping error" );
    TEAGN_TRUE_OR_THROW( ( newparams.max_imap_error_ >= 0 ),
      "Invalid maximum allowed inverse mapping error" );
       
    const unsigned int req_tie_pts_nmb = getMinRequiredTiePoints();

    if( newparams.tiepoints_.size() < req_tie_pts_nmb ) {
      return false;
    } else {
      internal_params_ = newparams;
      
      switch( newparams.out_rem_strat_ ) {
        case TeGTParams::NoOutRemotion :
        {
          if( computeParameters( internal_params_ ) ) {
            
            if( ( newparams.max_dmap_error_ >= 
              getDirectMappingError( internal_params_ ) ) &&
              ( newparams.max_imap_error_ >= 
              getInverseMappingError( internal_params_ ) ) ) {
              
              return true;
            }
          }   
        
          break;
        }
        case TeGTParams::ExaustiveOutRemotion :
        {
          if( internal_params_.enable_multi_thread_ ) {
            if( exaustiveOutRemotion( internal_params_,
              TeGetPhysProcNumber() - 1 ) ) {
              
              return true;
            }          
          } else {
            if( exaustiveOutRemotion( internal_params_, 0 ) ) {
              return true;
            }
          }
        
          break;
        }
        case TeGTParams::LWAOutRemotion :
        {
          if( LWAOutRemotion( internal_params_ ) ) {
            return true;
          }
                  
          break;
        }        
        default : 
        {
          TEAGN_LOG_AND_THROW( 
            "Invalid outliers remotion strategy" )
          break;
        }
      
      }
    }
  }
  
  internal_params_.reset();
  
  return false;
}


double TeGeometricTransformation::getDirectMappingError(  
  const TeGTParams& params ) const
{
  TEAGN_DEBUG_CONDITION( isTransDefined( params ),
    "Transformation not defined" )   
    
  unsigned int tiepoints_size = params.tiepoints_.size();
  
  double max_error = 0;
  double current_error = 0;
  
  for( unsigned int tpindex = 0 ; tpindex < tiepoints_size ; ++tpindex ) {
    current_error = getDirectMappingError( params.tiepoints_[ tpindex ], 
      params );
    
    if( current_error > max_error ) {
      max_error = current_error;
    }
  }
  
  return max_error;
}


double TeGeometricTransformation::getInverseMappingError(  
  const TeGTParams& params ) const
{
  TEAGN_DEBUG_CONDITION( isTransDefined( params ),
    "Transformation not defined" )  
    
  unsigned int tiepoints_size = params.tiepoints_.size();
  
  double max_error = 0;
  double current_error = 0;
  
  for( unsigned int tpindex = 0 ; tpindex < tiepoints_size ; ++tpindex ) {
    current_error = getInverseMappingError( params.tiepoints_[ tpindex ], 
      params );
    
    if( current_error > max_error ) {
      max_error = current_error;
    }
  }
  
  return max_error;
}


double TeGeometricTransformation::getDMapRMSE(  
  const TeGTParams& params ) const
{
  TEAGN_DEBUG_CONDITION( isTransDefined( params ),
    "Transformation not defined" )   
    
  unsigned int tiepoints_size = params.tiepoints_.size();
  
  if( tiepoints_size == 0 )
  {
    return 0;
  }
  else
  {
    double error2_sum = 0;
    double current_error = 0;
    
    for( unsigned int tpindex = 0 ; tpindex < tiepoints_size ; ++tpindex ) {
      current_error = getDirectMappingError( params.tiepoints_[ tpindex ], 
        params );
        
      error2_sum += ( current_error * current_error );
    }
    
    return sqrt( error2_sum / ( (double)tiepoints_size ) );
  }
}


double TeGeometricTransformation::getIMapRMSE(  
  const TeGTParams& params ) const
{
  TEAGN_DEBUG_CONDITION( isTransDefined( params ),
    "Transformation not defined" )   
    
  unsigned int tiepoints_size = params.tiepoints_.size();
  
  if( tiepoints_size == 0 )
  {
    return 0;
  }
  else
  {
    double error2_sum = 0;
    double current_error = 0;
    
    for( unsigned int tpindex = 0 ; tpindex < tiepoints_size ; ++tpindex ) {
      current_error = getInverseMappingError( params.tiepoints_[ tpindex ], 
        params );
        
      error2_sum += ( current_error * current_error );
    }
    
    return sqrt( error2_sum / ( (double)tiepoints_size ) );
  }  
}


double TeGeometricTransformation::getDirectMappingError( 
  const TeCoordPair& tie_point, const TeGTParams& params ) const
{
  TEAGN_DEBUG_CONDITION( isTransDefined( params ),
    "Transformation not defined" )   
    
  TeCoord2D direct_mapped_point;

  directMap( params, tie_point.pt1, direct_mapped_point );
    
  double diff_x = tie_point.pt2.x() - direct_mapped_point.x();
  double diff_y = tie_point.pt2.y() - direct_mapped_point.y();
    
  return hypot( diff_x, diff_y );
}


double TeGeometricTransformation::getInverseMappingError( 
  const TeCoordPair& tie_point, const TeGTParams& params ) const
{
  TEAGN_DEBUG_CONDITION( isTransDefined( params ),
    "Transformation not defined" )   
    
  TeCoord2D inverse_mapped_point;

  inverseMap( params, tie_point.pt2, inverse_mapped_point );
    
  double diff_x = tie_point.pt1.x() - inverse_mapped_point.x();
  double diff_y = tie_point.pt1.y() - inverse_mapped_point.y();
    
  return hypot( diff_x, diff_y );
}


bool TeGeometricTransformation::recombineSeed( std::vector<unsigned int>& seed, 
  const unsigned int& seedpos, const unsigned int& elements_nmb )
{
  unsigned int seed_size = seed.size();
  
  if( seedpos >= seed_size ) {
    return false;
  }

  if( seed[ seedpos ]  >= ( elements_nmb - seed_size + seedpos + 1 ) ) {
    if( seedpos == seed_size - 1 ) {
      return false;
    } else if( seedpos == 0 ) {
      return recombineSeed( seed, seedpos + 1, elements_nmb ) ;
    } else {
      return recombineSeed( seed, seedpos + 1, elements_nmb ) ;
    };
  } else if( seed[ seedpos ]  == 0 ) {
    if( seedpos == 0 ) {
      seed[ seedpos ] = 1 ;
      return recombineSeed( seed, seedpos + 1, elements_nmb );
    } else if( seedpos == seed_size - 1 ) {
      seed[ seedpos ] = seed[ seedpos - 1 ] + 1;
      return true;
    } else {
      seed[ seedpos ] = seed[ seedpos - 1 ] + 1;
      seed[ seedpos + 1 ] = 0;
      return recombineSeed( seed, seedpos + 1, elements_nmb );
    }
  } else {
    if( seedpos == seed_size - 1 ) {
      seed[ seedpos ] = seed[ seedpos ] + 1;
      return true;
    } else if( seedpos == 0 ) {
      if( recombineSeed( seed, seedpos + 1, elements_nmb ) ) {
        return true;
      } else {
        seed[ seedpos ] = seed[ seedpos ] + 1;
        seed[ seedpos + 1 ] = 0;
        return recombineSeed( seed, seedpos + 1, elements_nmb );
      }
    } else {
      if( recombineSeed( seed, seedpos + 1, elements_nmb ) ) {
        return true;
      } else {
        seed[ seedpos ] = seed[ seedpos ] + 1;
        seed[ seedpos + 1 ] = 0;
        return recombineSeed( seed, seedpos + 1, elements_nmb );
      }
    }
  }
}


TeGeometricTransformation* TeGeometricTransformation::DefaultObject( 
  const TeGTParams& )
{ 
  TEAGN_LOG_AND_THROW( "Trying to create an invalid "
    "TeGemetricTransformation instance" );
  
  return 0;
}; 


bool TeGeometricTransformation::exaustiveOutRemotion( 
  TeGTParams& params, unsigned int threads_nmb )
{
  TEAGN_DEBUG_CONDITION( ( params.out_rem_strat_ == 
    TeGTParams::ExaustiveOutRemotion ), 
    "Inconsistent outliers remotion strategy" )

  /* Initiating seed */
  
  std::vector<unsigned int> comb_seed_vec;       
    
  unsigned int req_tie_pts_nmb = getMinRequiredTiePoints();
  
  for( unsigned int comb_seed_vec_index = 0 ; 
    comb_seed_vec_index < req_tie_pts_nmb ;
    ++comb_seed_vec_index ) {
    
    comb_seed_vec.push_back( 0 );
  }
  
  /* initializing mutexes */
  
  TeMutex comb_seed_vec_mutex;
  TeMutex trans_params_mutex;
  
  /* Initializing threads */
  
  TeGTParams best_trans_params;
  double best_trans_dmap_error = DBL_MAX;
  double best_trans_imap_error = DBL_MAX;
  
  std::vector< TeThreadFunctor::pointer > threads_vector;
  
  TeThreadParameters thread_params;
  
  thread_params.store( "req_tie_pts_nmb", req_tie_pts_nmb );  
  thread_params.store( "init_trans_params_ptr", 
    (TeGTParams const*)(&params) );
  thread_params.store( "best_trans_params_ptr", 
    &best_trans_params );    
  thread_params.store( "best_trans_dmap_error_ptr", 
    &best_trans_dmap_error );      
  thread_params.store( "best_trans_imap_error_ptr", 
    &best_trans_imap_error );     
  thread_params.store( "comb_seed_vec_ptr", &comb_seed_vec );
  thread_params.store( "trans_params_mutex_ptr", 
    &trans_params_mutex );
  thread_params.store( "comb_seed_vec_mutex_ptr", 
    &comb_seed_vec_mutex );  
  thread_params.store( "geo_trans_ptr", 
    (TeGeometricTransformation const*)this );        
    
  unsigned int thread_index = 0;
  
  for( thread_index = 0 ; thread_index < threads_nmb ; 
    ++thread_index ) {
    
    TeThreadFunctor::pointer aux_thread_ptr( 
      new TeThreadFunctor( eORThreadEntry ) );
    aux_thread_ptr->setParameters( thread_params );
    aux_thread_ptr->start();
  
    threads_vector.push_back( aux_thread_ptr );
  }
  
  bool my_return = eORThreadEntry( thread_params );
  bool threads_return = true;
  
  for( thread_index = 0 ; thread_index < threads_nmb ; 
    ++thread_index ) {
    
    threads_vector[ thread_index ]->waitToFinish();
    
    threads_return = threads_return & 
      threads_vector[ thread_index ]->getReturnValue();
  }  
  
  if( my_return & threads_return ) {
    if( best_trans_params.tiepoints_.size() >= req_tie_pts_nmb ) {
      params = best_trans_params;
      
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }
}


bool TeGeometricTransformation::eORThreadEntry( 
  const TeThreadParameters& params )
{
  /* Extracting parameters */
  
  unsigned int req_tie_pts_nmb = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "req_tie_pts_nmb",
    req_tie_pts_nmb ), "Missing parameter" );
    
  TeGTParams const* init_trans_params_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "init_trans_params_ptr",
    init_trans_params_ptr ), "Missing parameter" );
  
  TeGTParams* best_trans_params_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "best_trans_params_ptr",
    best_trans_params_ptr ), "Missing parameter" );   
    
  double* best_trans_dmap_error_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "best_trans_dmap_error_ptr",
    best_trans_dmap_error_ptr ), "Missing parameter" );     
    
  double* best_trans_imap_error_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "best_trans_imap_error_ptr",
    best_trans_imap_error_ptr ), "Missing parameter" );     

  std::vector<unsigned int>* comb_seed_vec_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "comb_seed_vec_ptr",
    comb_seed_vec_ptr ), "Missing parameter" );
    
  TeMutex* trans_params_mutex_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "trans_params_mutex_ptr",
    trans_params_mutex_ptr ), "Missing parameter" ); 

  TeMutex* comb_seed_vec_mutex_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "comb_seed_vec_mutex_ptr",
    comb_seed_vec_mutex_ptr ), "Missing parameter" );  
    
  TeGeometricTransformation const* geo_trans_ptr = 0;
  TEAGN_TRUE_OR_THROW( params.retrive( "geo_trans_ptr",
    geo_trans_ptr ), "Missing parameter" ); 
  
  /* Optimized local variables based on the input parameters */
  
  std::vector<unsigned int>& comb_seed_vec =
    (*comb_seed_vec_ptr);  
    
  TeMutex& comb_seed_vec_mutex = (*comb_seed_vec_mutex_ptr); 
  
  const TeGeometricTransformation& geo_trans = (*geo_trans_ptr); 
  
  /* Copying some parameters to local variables to avoid
     mutex overhead */
     
  trans_params_mutex_ptr->lock();
     
  const TeGTParams initial_trans_params = (*init_trans_params_ptr);     
  const unsigned int initial_tiepoints_size = 
    initial_trans_params.tiepoints_.size();   
  
  trans_params_mutex_ptr->unLock();
        
  /* Trying to find the best tie-points by building 
    the transformation with the highest number of
    tie-points, but with an acceptable mapping error */
    
  TeGTParams local_best_params;
  double local_best_params_dmap_error = DBL_MAX;
  double local_best_params_imap_error = DBL_MAX;
  
  TeGTParams curr_params = initial_trans_params;
  double curr_params_dmap_error = DBL_MAX;
  double curr_params_imap_error = DBL_MAX; 
  
  TeGTParams cp_plus_new_point;
  double cp_plus_new_point_dmap_error = DBL_MAX;
  double cp_plus_new_point_imap_error = DBL_MAX;
  
  unsigned int tiepoints_index = 0;
  bool point_already_present = false;
  bool seed_recombined = true;
  unsigned int comb_seed_vec_index_2 = 0;
  unsigned int comb_seed_vec_index_3 = 0;
  
  while( seed_recombined ) {
    /* trying to recombine seed */
    
    comb_seed_vec_mutex.lock();
    seed_recombined = recombineSeed( comb_seed_vec, 0, 
      initial_tiepoints_size );
    comb_seed_vec_mutex.unLock();
      
    if( seed_recombined ) {
      /* Extracting tie-points from the original vector */
      
      curr_params.tiepoints_.clear();
      
      for( comb_seed_vec_index_2 = 0 ; 
        comb_seed_vec_index_2 < req_tie_pts_nmb ;
        ++comb_seed_vec_index_2 ) {
        
        curr_params.tiepoints_.push_back( 
          initial_trans_params.tiepoints_[ comb_seed_vec[ 
          comb_seed_vec_index_2 ] - 1 ] );
      }
      
      /* Trying to generate a valid transformation */
      
      if( geo_trans.computeParameters( curr_params ) ) {
        curr_params_dmap_error = geo_trans.getDirectMappingError( 
          curr_params );
        curr_params_imap_error = geo_trans.getInverseMappingError( 
          curr_params );
      
        if( ( initial_trans_params.max_dmap_error_ >= 
          curr_params_dmap_error ) &&
          ( initial_trans_params.max_imap_error_ >= 
          curr_params_imap_error ) ) {
        
          /* Trying to insert more tie-points into current 
              transformation */
          
          for( tiepoints_index = 0 ; tiepoints_index < 
            initial_tiepoints_size ; ++tiepoints_index ) {
            
            /* Verifying if the current tie-point is already 
               present */
            
            point_already_present = false;
            
            for( comb_seed_vec_index_3 = 0 ; 
              comb_seed_vec_index_3 < req_tie_pts_nmb ;
              ++comb_seed_vec_index_3 ) {
            
              if( tiepoints_index == ( 
                comb_seed_vec[ comb_seed_vec_index_3 ] - 1 ) ) {
              
                point_already_present = true;
                break;
              }
            }
            
            if( ! point_already_present ) {
              cp_plus_new_point = curr_params;
              
              cp_plus_new_point.tiepoints_.push_back( 
                initial_trans_params.tiepoints_[ tiepoints_index ] );
              
              /* Verifying if the new tie-point insertion does 
                not generate an invalid transformation */
              
              if( geo_trans.computeParameters( cp_plus_new_point ) ) {
                cp_plus_new_point_dmap_error =
                  geo_trans.getDirectMappingError( cp_plus_new_point );
                cp_plus_new_point_imap_error =
                  geo_trans.getInverseMappingError( cp_plus_new_point );
                  
                if( ( cp_plus_new_point_dmap_error <=
                  initial_trans_params.max_dmap_error_ ) &&
                  ( cp_plus_new_point_imap_error <=
                  initial_trans_params.max_imap_error_ ) ) {
                  
                  curr_params = cp_plus_new_point;
                  curr_params_dmap_error = 
                    cp_plus_new_point_dmap_error;
                  curr_params_imap_error =
                    cp_plus_new_point_imap_error;  
                }
              }
            }
          }
          
          /* A valid transformation was generated, now 
            verifying 
            if the number of tie-poits is greater then the current 
            thread local best transformation.
          */            
          
          if( curr_params.tiepoints_.size() >= 
            local_best_params.tiepoints_.size() ) {
            
            if( ( curr_params_dmap_error < 
              local_best_params_dmap_error ) &&
              ( curr_params_imap_error <
                local_best_params_imap_error ) ) {
            
              local_best_params = curr_params;
              local_best_params_dmap_error = curr_params_dmap_error;
              local_best_params_imap_error = curr_params_imap_error;
            }
          }           
        }
      } //if( geo_trans.computeParameters( curr_params ) )
    } //if( seed_recombined )
  } //while( seed_recombined )
  
  /* A valid local thread transformation was generated, now 
    verifying 
    if the error is smaller then the current 
    global transformation.
  */

  if( local_best_params.tiepoints_.size() >= req_tie_pts_nmb ) {
    trans_params_mutex_ptr->lock();
    
    if( local_best_params.tiepoints_.size() >=
      best_trans_params_ptr->tiepoints_.size() ) {
      
      if( ( local_best_params_dmap_error < 
        (*best_trans_dmap_error_ptr) ) &&
        ( local_best_params_imap_error < 
        (*best_trans_imap_error_ptr) ) ) {
      
        (*best_trans_params_ptr) = local_best_params;
        (*best_trans_dmap_error_ptr) = local_best_params_dmap_error;
        (*best_trans_imap_error_ptr) = local_best_params_imap_error;       
      }      
    }
    
    trans_params_mutex_ptr->unLock();
  }
  
  return true;
}


bool TeGeometricTransformation::LWAOutRemotion( 
  TeGTParams& external_params )
{
  TEAGN_DEBUG_CONDITION( ( external_params.out_rem_strat_ == 
    TeGTParams::LWAOutRemotion ), 
    "Inconsistent outliers remotion strategy" )
    
  const unsigned int req_tie_pts_nmb = getMinRequiredTiePoints();
  
  if( external_params.tiepoints_.size() == req_tie_pts_nmb ) {
    return computeParameters( external_params );
  } else if( external_params.tiepoints_.size() > req_tie_pts_nmb ) {
    /* Global vars */
    
    const double max_dmap_error = 
      external_params.max_dmap_error_;
    const double max_imap_error = 
      external_params.max_imap_error_;
    const double max_dmap_rmse = 
      external_params.max_dmap_rmse_;
    const double max_imap_rmse = 
      external_params.max_imap_rmse_;
          
    /* Computing the initial global transformation */
    
    if( ! computeParameters( external_params ) ) {
      return false;
    }    
    
    if( ( getDirectMappingError( external_params ) 
      <= max_dmap_error ) && 
      ( getInverseMappingError( external_params ) 
      <= max_imap_error ) &&
      ( getDMapRMSE( external_params ) 
      <= max_dmap_rmse ) && 
      ( getIMapRMSE( external_params ) 
      <= max_imap_rmse ) ) {
    
      /* This transformation has no outliers */
      
      return true;
    }    
    
    /* Iterating over the current transformation tie-points */
    
    TeGTParams best_params;
    double best_params_dmap_rmse = DBL_MAX;
    double best_params_imap_rmse = DBL_MAX;
    double best_params_dmap_error = DBL_MAX;
    double best_params_imap_error = DBL_MAX;
    
    bool transformation_not_updated = false;
    unsigned int iterations_remainning = (unsigned int)
      external_params.tiepoints_.size();
    std::vector< TPDataNode > norm_err_vec;     
    std::list< ExcTPDataNode > exc_tp_list;
    TeGTParams iteration_params = external_params;
  
    while( ( iteration_params.tiepoints_.size() > 
      req_tie_pts_nmb ) && iterations_remainning )
    {
      unsigned int iter_tps_nmb = (unsigned int)
        iteration_params.tiepoints_.size();
      transformation_not_updated = true;
      
      /* Updating the normalized error map */
      
      updateTPErrVec( iteration_params, norm_err_vec );
      
      /* Generating all possible transformations without 
         each tie-point, starting with the worse point */
      
      for( int norm_err_vec_idx = ((int)norm_err_vec.size()) - 1 ;
        ( norm_err_vec_idx > ( -1 ) ) ; --norm_err_vec_idx )
      {
        const unsigned int& cur_candtp_idx = 
          norm_err_vec[ norm_err_vec_idx ].tpindex_;
          
        TEAGN_DEBUG_CONDITION( ( cur_candtp_idx < 
          iteration_params.tiepoints_.size() ), "Invalid index" )

        /* Generating a transformation parameters without the
          current tie-point (bigger error)*/
            
        TeGTParams new_iteration_params = iteration_params;
        new_iteration_params.tiepoints_.clear();
        new_iteration_params.direct_parameters_.Clear();
        new_iteration_params.inverse_parameters_.Clear();
            
        for( unsigned int tpindex2 = 0 ; tpindex2 < iter_tps_nmb ; 
          ++tpindex2 ) 
        {
          if( cur_candtp_idx != tpindex2 ) {
            new_iteration_params.tiepoints_.push_back( 
              iteration_params.tiepoints_[ tpindex2 ] );
          }
        }            
      
        /* Trying to generate a transformation without the current
            candidate tie-point for exclusion */      
        
        if( computeParameters( new_iteration_params ) ) 
        {
          double new_it_dmap_rmse = 
            getDMapRMSE( new_iteration_params );
          double new_it_imap_rmse = 
            getIMapRMSE( new_iteration_params );
          
          if( ( best_params_dmap_rmse > new_it_dmap_rmse ) && 
            ( best_params_imap_rmse > new_it_imap_rmse ) ) 
          {
            double new_it_dmap_error = 
              getDirectMappingError( new_iteration_params );
            double new_it_imap_error = 
              getInverseMappingError( new_iteration_params );
                        
            TeCoordPair excluded_tp = 
              iteration_params.tiepoints_[ cur_candtp_idx ];
              
            /* Trying to insert back other tie-points excluded
               before */
            
            if( exc_tp_list.size() > 0 )
            {
              /* Updating the excluded tie points errors map */
              
              updateExcTPErrList( new_iteration_params, 
                exc_tp_list );
                
              /* Iterating over the excluded tps */
              
              std::list< ExcTPDataNode >::iterator 
                exc_tps_list_it = exc_tp_list.begin();
              std::list< ExcTPDataNode >::iterator 
                exc_tps_list_it_end = exc_tp_list.end();
                
              while( exc_tps_list_it != exc_tps_list_it_end )
              {
                const ExcTPDataNode& curr_exc_tp_data = 
                  *exc_tps_list_it;

                /* Does this tp has direct and inverse errors 
                   smaller than the current new_iteration_params ?? */
                        
                if( ( curr_exc_tp_data.dmap_error_ <= 
                  new_it_dmap_error ) && 
                  ( curr_exc_tp_data.imap_error_ <= 
                  new_it_imap_error ) )
                {
                  /* Trying to generate a trasformation with
                     this point */
                     
                  TeGTParams new_iteration_params_plus1tp = 
                    new_iteration_params;
                  new_iteration_params.tiepoints_.push_back(
                    curr_exc_tp_data.tp_ );
                    
                  if( computeParameters( 
                    new_iteration_params_plus1tp ) ) 
                  {
                    double newp1tp_dmap_rmse = getDMapRMSE( 
                      new_iteration_params_plus1tp );
                    double newp1tp_imap_rmse = getIMapRMSE( 
                      new_iteration_params_plus1tp );
                    
                    if( ( new_it_dmap_rmse >= newp1tp_dmap_rmse ) && 
                      ( new_it_imap_rmse >= newp1tp_imap_rmse ) ) 
                    {
                      new_iteration_params = 
                        new_iteration_params_plus1tp;
                      new_it_dmap_error = getDirectMappingError( 
                        new_iteration_params_plus1tp );
                      new_it_imap_error = getInverseMappingError( 
                        new_iteration_params_plus1tp ); 
                      new_it_dmap_rmse = newp1tp_dmap_rmse;
                      new_it_imap_rmse = newp1tp_imap_rmse;
                        
                      exc_tp_list.erase( exc_tps_list_it );
                        
                      updateExcTPErrList( new_iteration_params, 
                        exc_tp_list );   
                        
                      exc_tps_list_it = exc_tp_list.begin();
                      exc_tps_list_it_end = exc_tp_list.end();                    
                    }
                    else
                    {
                      ++exc_tps_list_it;
                    }
                  } 
                  else
                  {
                    ++exc_tps_list_it;
                  }                   
                }
                else
                {  
                  ++exc_tps_list_it;
                }
              }
            }
            
            /* Updating the best transformation parameters */
            
            best_params_dmap_error = new_it_dmap_error;
            best_params_imap_error = new_it_imap_error;
            best_params_dmap_rmse = new_it_dmap_rmse;
            best_params_imap_rmse = new_it_imap_rmse;
            best_params = new_iteration_params;
            
            /* Updating the next iteration parameters */
            
            iteration_params = new_iteration_params;
            
            transformation_not_updated = false;
            
            /* Putting the excluded tie-point into the 
               excluded tie-points map */
               
            ExcTPDataNode exctpdata;
            exctpdata.tp_ = excluded_tp;
            exc_tp_list.push_back( exctpdata );
            
            /* break the current tie-points loop */
            
            break;
          } //if( ( best_params_dmap_error > new_it_dmap_error ) && 
        } //if( computeParameters( new_iteration_params ) ) {    
      }
      
      if( transformation_not_updated ) {
        /* There is no way to improve the current transformation
           since all tie-points were tested */
        
        break; /* break the iterations loop */
      } else {
        if( ( max_dmap_error >= best_params_dmap_error ) && 
            ( max_imap_error >= best_params_imap_error ) &&
            ( max_dmap_rmse >= best_params_dmap_rmse ) && 
            ( max_imap_rmse >= best_params_imap_rmse ) ) {
            
          /* A valid transformation was found */
          break;
        }
      }
      
      --iterations_remainning;
    }//while( params.tiepoints_.size() > req_tie_pts_nmb ) {
    
    if( ( best_params.tiepoints_.size() >= req_tie_pts_nmb ) &&
        ( max_dmap_error >= best_params_dmap_error ) && 
        ( max_imap_error >= best_params_imap_error ) &&
        ( max_dmap_rmse >= best_params_dmap_rmse ) && 
        ( max_imap_rmse >= best_params_imap_rmse ) ) {
        
      external_params = best_params;
      
      return true;
    } else {
      return false;
    }    
  }
  
  return false;
}


void TeGeometricTransformation::updateExcTPErrList( 
  const TeGTParams& params,
  std::list< ExcTPDataNode >& exc_tp_list )
  const
{
  if( exc_tp_list.size() > 0 )
  {
    /* Updating the new direct and inverse mapping errors */
     
    std::list< ExcTPDataNode >::iterator tp_list_it = 
      exc_tp_list.begin();
    std::list< ExcTPDataNode >::iterator tp_list_it_end = 
      exc_tp_list.end();
      
    double dmap_err_min =  DBL_MAX;
    double dmap_err_max =  (-1.0) * DBL_MAX;
    double imap_err_min =  DBL_MAX;  
    double imap_err_max =  (-1.0) * DBL_MAX;
    
    while( tp_list_it != tp_list_it_end )
    {
      ExcTPDataNode& curr_node = *tp_list_it;
      
      const TeCoordPair& cur_ex_tp = curr_node.tp_;
      
      curr_node.dmap_error_ = getDirectMappingError( cur_ex_tp, 
        params );
      curr_node.imap_error_ = getInverseMappingError( cur_ex_tp, 
        params );        
        
      if( dmap_err_min > curr_node.dmap_error_ ) {
        dmap_err_min = curr_node.dmap_error_;
      }
      if( dmap_err_max < curr_node.dmap_error_ ) {
        dmap_err_max = curr_node.dmap_error_;
      }        
        
      if( imap_err_min > curr_node.imap_error_ ) {
        imap_err_min = curr_node.imap_error_;
      }
      if( imap_err_max < curr_node.imap_error_ ) {
        imap_err_max = curr_node.imap_error_;
      }            
      
      ++tp_list_it;
    }
    
    double dmap_err_range = dmap_err_max - dmap_err_min;
    double imap_err_range = imap_err_max - imap_err_min;
    
    /* Updating the normalized error */
    
    tp_list_it = exc_tp_list.begin();
    
    if( ( dmap_err_range == 0.0 ) &&
      ( imap_err_range == 0.0 ) )
    {
      while( tp_list_it != tp_list_it_end )
      {
        tp_list_it->norm_error_ = 0.0;
        
        ++tp_list_it;
      }
    }
    else if( dmap_err_range == 0.0 ) 
    {
      while( tp_list_it != tp_list_it_end )
      {
        tp_list_it->norm_error_ =
          ( 
            ( 
              (  
                tp_list_it->imap_error_ - imap_err_min              
              ) 
              / imap_err_range 
            ) 
          );
        
        ++tp_list_it;
      }
    }
    else if( imap_err_range == 0.0 ) 
    {
      while( tp_list_it != tp_list_it_end )
      {
        tp_list_it->norm_error_ =
          ( 
            ( 
              ( 
                tp_list_it->dmap_error_ - dmap_err_min
              ) 
              / dmap_err_range 
            ) 
          );
        
        ++tp_list_it;
      }
    }
    else
    {
      while( tp_list_it != tp_list_it_end )
      {
        tp_list_it->norm_error_ =
          ( 
            ( 
              ( 
                tp_list_it->dmap_error_ - dmap_err_min
              ) 
              / dmap_err_range 
            ) 
            + 
            ( 
              (  
                tp_list_it->imap_error_ - imap_err_min              
              ) 
              / imap_err_range 
            ) 
          );
        
        ++tp_list_it;
      }
    }
    
    /* Sorting list */
    
    exc_tp_list.sort();
    
    /* printing */
/*   
    tp_list_it = exc_tp_list.begin();
    tp_list_it_end = exc_tp_list.end();
    std::cout << std::endl << "Excluded Tie points" << std::endl;
    while( tp_list_it != tp_list_it_end )
    {
      const ExcTPDataNode& node = *tp_list_it;
      
      std::cout << node.norm_error_ << " " <<
       " [" + Te2String( node.tp_.pt1.x(),2 ) + "," +
       Te2String( node.tp_.pt1.y(),2 ) + "-" +
       Te2String( node.tp_.pt2.x(),2 ) + "," +
       Te2String( node.tp_.pt2.y(),2 ) + "]"
       << std::endl;      
      
      ++tp_list_it;
    }    
    std::cout << std::endl;
*/  
  }
}


void TeGeometricTransformation::updateTPErrVec( 
  const TeGTParams& params, 
  std::vector< TPDataNode >& errvec ) const
{
  /* Calculating the two mapping errors */
  
  const unsigned int iter_tps_nmb = (unsigned int)
    params.tiepoints_.size();  
    
  errvec.clear();

  double dmap_err_vec_min = DBL_MAX;
  double dmap_err_vec_max = ( -1.0 ) * DBL_MAX;
  double imap_err_vec_min = DBL_MAX;
  double imap_err_vec_max = ( -1.0 ) * DBL_MAX;      
  TPDataNode dummy_struct;
  dummy_struct.norm_error_ = 0.0;
  
  for( unsigned int par_tps_vec_idx = 0 ; par_tps_vec_idx < iter_tps_nmb ; 
    ++par_tps_vec_idx ) 
  {
    const TeCoordPair& cur_tp = params.tiepoints_[ 
      par_tps_vec_idx ];
      
    dummy_struct.tpindex_ = par_tps_vec_idx;
    dummy_struct.dmap_error_ = getDirectMappingError( cur_tp, 
      params );
    dummy_struct.imap_error_ = getInverseMappingError( cur_tp, 
      params );
    
    errvec.push_back( dummy_struct );
    
    if( dmap_err_vec_min > dummy_struct.dmap_error_ ) {
      dmap_err_vec_min = dummy_struct.dmap_error_;
    }
    if( dmap_err_vec_max < dummy_struct.dmap_error_ ) {
      dmap_err_vec_max = dummy_struct.dmap_error_;
    }        
    
    if( imap_err_vec_min > dummy_struct.imap_error_ ) {
      imap_err_vec_min = dummy_struct.imap_error_;
    }
    if( imap_err_vec_max < dummy_struct.imap_error_ ) {
      imap_err_vec_max = dummy_struct.imap_error_;
    }            
  }
  
  double dmap_err_vec_range = 
    dmap_err_vec_max - dmap_err_vec_min;
  double imap_err_vec_range = 
    imap_err_vec_max - imap_err_vec_min;
  
  if( ( dmap_err_vec_range != 0.0 ) ||
    ( imap_err_vec_range != 0.0 ) )
  {
    /* Updating normalized mapping errors */
    
    if( dmap_err_vec_range == 0.0 )
    {
      for( unsigned int errvec_idx = 0 ; errvec_idx < iter_tps_nmb ; 
        ++errvec_idx ) 
      {
        TPDataNode& curr_elem = errvec[ errvec_idx ];
        
        curr_elem.norm_error_ = 
          ( 
            (  
              curr_elem.imap_error_ - 
              imap_err_vec_min              
            ) 
            / imap_err_vec_range 
          );
      }
    }
    else if( imap_err_vec_range == 0.0 )
    {
      for( unsigned int errvec_idx = 0 ; errvec_idx < iter_tps_nmb ; 
        ++errvec_idx ) 
      {
        TPDataNode& curr_elem = errvec[ errvec_idx ];
        
        curr_elem.norm_error_ = 
          ( 
            ( 
              curr_elem.dmap_error_ - 
              dmap_err_vec_min
            ) 
            / dmap_err_vec_range 
          );
      }
    }
    else
    {
      for( unsigned int errvec_idx = 0 ; errvec_idx < iter_tps_nmb ; 
        ++errvec_idx ) 
      {
        TPDataNode& curr_elem = errvec[ errvec_idx ];
        
        curr_elem.norm_error_ = 
          ( 
            ( 
              curr_elem.dmap_error_ - 
              dmap_err_vec_min
            ) 
            / dmap_err_vec_range 
          ) 
          + 
          ( 
            (  
              curr_elem.imap_error_ - 
              imap_err_vec_min              
            ) 
            / imap_err_vec_range 
          );
      }
    }
    
    /* Sorting */
    
    sort( errvec.begin(), errvec.end() );

    /* printing */
/*  
    std::cout << std::endl << "Tie points" << std::endl;
    for( unsigned int errvec_idx = 0 ; errvec_idx < iter_tps_nmb ; 
      ++errvec_idx ) 
    {
      const unsigned int& tp_pars_vec_idx = errvec[ errvec_idx ].tpindex_;
      const TeCoordPair& currtp = params.tiepoints_[ 
      tp_pars_vec_idx ];
      
      std::cout << errvec[ errvec_idx ].norm_error_ << " " <<
       " [" + Te2String( currtp.pt1.x(),2 ) + "," +
       Te2String( currtp.pt1.y(),2 ) + "-" +
       Te2String( currtp.pt2.x(),2 ) + "," +
       Te2String( currtp.pt2.y(),2 ) + "]"
       << std::endl;
    }    
*/    
  }
  
  
}

