#include "TePDIJointHistogram.hpp"

#include "TePDIPIManager.hpp"

#include <TeOverlay.h>
#include <TeBox.h>
#include <TeGeometry.h>
#include <TeAgnostic.h>

TePDIJointHistogram::TePDIJointHistogram()
{
  progressEnabled_ = true;
}


TePDIJointHistogram::TePDIJointHistogram( const TePDIJointHistogram& external )
{
  operator=( external );
}


TePDIJointHistogram::~TePDIJointHistogram()
{
}

const TePDIJointHistogram& TePDIJointHistogram::operator=( const 
  TePDIJointHistogram& external )    
{
  jHistContainer_ = external.jHistContainer_;
  r1Histogram_ = external.r1Histogram_;
  r2Histogram_ = external.r2Histogram_;
  
  return external;
}

void TePDIJointHistogram::toggleProgress( bool enabled )
{
  progressEnabled_ = enabled;
}

bool TePDIJointHistogram::update( TeRaster& inputRaster1, 
  unsigned int inputRasterChn1, TeRaster& inputRaster2, 
  unsigned int inputRasterChn2, TeStrategicIterator iterStrat, 
  unsigned int levels, const TePolygonSet& restricPolSet )
{
  TEAGN_TRUE_OR_RETURN( inputRaster1.params().status_ != 
    TeRasterParams::TeNotReady, "Raster1 not ready" )
  TEAGN_TRUE_OR_RETURN( inputRaster2.params().status_ != 
    TeRasterParams::TeNotReady, "Raster2 not ready" )
    
  TEAGN_TRUE_OR_RETURN( inputRaster1.params().resx_ ==
    inputRaster2.params().resx_, "X resolution mismatch" )
  TEAGN_TRUE_OR_RETURN( inputRaster1.params().resy_ ==
    inputRaster2.params().resy_, "Y resolution mismatch" )
      
  clear();
    
  // Calc the intersection between the restriction polygon set,
  // and both rasters bounding box
     
  TePolygonSet interPS;
  {
    TePolygon pol1 = polygonFromBox( inputRaster1.params().boundingBox() );
    TePolygon pol2 = polygonFromBox( inputRaster2.params().boundingBox() );
    
    TePolygonSet ps1;
    ps1.add( pol1 );
    
    TePolygonSet ps2;
    ps2.add( pol2 );
    
    TePolygonSet boxesInterPS;
    TEAGN_TRUE_OR_RETURN( TeOVERLAY::TeIntersection( ps1, ps2, boxesInterPS ),
      "Unable to find the intersection between both raster bounding boxes" );
    
    if( restricPolSet.size() == 0 )
    {
      interPS = boxesInterPS;
    }
    else
    {
      TEAGN_TRUE_OR_RETURN( TeOVERLAY::TeIntersection( boxesInterPS, restricPolSet, 
        interPS ), "Unable to find the intersection between both raster bounding boxes" );
    }
  }
    
  // Pallete based rasters not supported
    
  if( ( inputRaster1.params().photometric_[ inputRasterChn1 ] == 
    TeRasterParams::TePallete ) ||
    ( inputRaster2.params().photometric_[ inputRasterChn2 ] == 
    TeRasterParams::TePallete ) ) 
  {
    TEAGN_LOG_AND_RETURN( "Pallete based rasters not supported" )
  }
    
  // Optimization for floating point rasters

  if( ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeDOUBLE ) ||
      ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeFLOAT ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeDOUBLE ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeFLOAT ) )
  {
    return updateFloat( inputRaster1, inputRasterChn1, inputRaster2, 
      inputRasterChn2, iterStrat, levels, interPS );
  }
  
  // Optmization for integer rasters
		  
  if( ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeINTEGER ) ||
      ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeUNSIGNEDLONG ) ||
      ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeLONG ) ||
      ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeSHORT ) ||
      ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeUNSIGNEDSHORT ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeINTEGER ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeUNSIGNEDLONG ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeLONG ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeSHORT ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeUNSIGNEDSHORT ) )
  {
    return updateInteger( inputRaster1, inputRasterChn1, inputRaster2, 
      inputRasterChn2, iterStrat, levels, interPS );
  }  
  
  // Optimization for 8 bits or 16 bits
  
  if( ( ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeCHAR ) ||
      ( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeUNSIGNEDCHAR ) ) &&
      ( ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeCHAR ) ||
      ( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeUNSIGNEDCHAR ) ) )
  {
    return update8Bits( inputRaster1, inputRasterChn1, inputRaster2, 
      inputRasterChn2, iterStrat, levels, interPS );
  }  
  
  TEAGN_LOG_AND_THROW( "Invalid data type" );
  return false;
}

void TePDIJointHistogram::clear()
{
  jHistContainer_.clear();
  r1Histogram_.clear();
  r2Histogram_.clear();
}

const TePDIHistogram& TePDIJointHistogram::getRaster1Hist() const
{
  return r1Histogram_;
}

const TePDIHistogram& TePDIJointHistogram::getRaster2Hist() const
{
  return r2Histogram_;
}

TePDIJointHistogram::const_iterator TePDIJointHistogram::begin() const
{
  return jHistContainer_.begin();
}

TePDIJointHistogram::const_iterator TePDIJointHistogram::end() const
{
  return jHistContainer_.end();
}

unsigned int TePDIJointHistogram::size() const
{
  return (unsigned int)jHistContainer_.size();
}

bool TePDIJointHistogram::updateFloat( TeRaster& inputRaster1, 
  unsigned int inputRasterChn1, TeRaster& inputRaster2, 
  unsigned int inputRasterChn2, TeStrategicIterator iterStrat, 
  unsigned int levels, const TePolygonSet& restricPolSet )
{
  TEAGN_NOT_IMPLEMENTED;
}

bool TePDIJointHistogram::updateInteger( TeRaster& inputRaster1, 
  unsigned int inputRasterChn1, TeRaster& inputRaster2, 
  unsigned int inputRasterChn2, TeStrategicIterator iterStrat, 
  unsigned int levels, const TePolygonSet& restricPolSet )
{
  TEAGN_NOT_IMPLEMENTED;
}

bool TePDIJointHistogram::update8Bits( TeRaster& inputRaster1, 
  unsigned int inputRasterChn1, TeRaster& inputRaster2, 
  unsigned int inputRasterChn2, TeStrategicIterator iterStrat, 
  unsigned int levels, const TePolygonSet& restricPolSet )
{
   /* Dummy use check */
   
   bool r1_no_dummy = ! inputRaster1.params().useDummy_;
   double r1_dummy = 0;
   
   if( ! r1_no_dummy ) {
     r1_dummy = inputRaster1.params().dummy_[ inputRasterChn1 ];
   }      
   
   bool r2_no_dummy = ! inputRaster2.params().useDummy_;
   double r2_dummy = 0;
   
   if( ! r2_no_dummy ) {
     r2_dummy = inputRaster2.params().dummy_[ inputRasterChn2 ];
   }       
     
   /* Defining level offset based on data bype */
   
   unsigned int level_offset1 = 0;
   
   if( inputRaster1.params().dataType_[ inputRasterChn1 ] == TeCHAR ) {
     level_offset1 = 127; /*( 256 / 2 ) - 1 */
   }
   
   unsigned int level_offset2 = 0;
   
   if( inputRaster2.params().dataType_[ inputRasterChn2 ] == TeCHAR ) {
     level_offset2 = 127; /*( 256 / 2 ) - 1 */
   }   
     
   /* Building initial histograms */
   
   unsigned int init_jhist[ 256 ][ 256 ];
   unsigned int init_r1hist[ 256 ];
   unsigned int init_r2hist[ 256 ];
   {
     unsigned int hist_index1 = 0;
     unsigned int hist_index2 = 0;
     
     for( hist_index1 = 0 ; hist_index1 < 256 ; ++hist_index1 )
     {
       init_r1hist[ hist_index1 ] = 0;
       init_r2hist[ hist_index1 ] = 0;
       
       for( hist_index2 = 0 ; hist_index2 < 256 ; ++hist_index2 )
         init_jhist[ hist_index1 ][ hist_index2 ] = 0;
     }
   }
   
   /* Analysing raster */
   
   {
     const unsigned long int progress_steps = 
       getProgressSteps( restricPolSet, inputRaster1.params().resy_ ); 
     
     TePDIPIManager progress( "Generating histogram", progress_steps, 
       progressEnabled_ );
   
     TeRaster::iteratorPoly r1_it;
     TeRaster::iteratorPoly r2_it;
     
     long int r1_line = 0;
     long int r1_col = 0;
     long int last_r1_line = 0;
     
     long int r2_line_off = 0;
     long int r2_col_off = 0;
     
     double r1_value_double = 0;
     double r2_value_double = 0;
     int r1_value_int = 0;
     int r2_value_int = 0;
     
     bool gotvalue1 = false;
     bool gotvalue2 = false;
     
     for( unsigned int polset_index = 0 ; polset_index < 
       restricPolSet.size() ; ++polset_index ) 
     {
       r1_it = inputRaster1.begin( restricPolSet[ polset_index ], 
         iterStrat, 0 );
       r2_it = inputRaster2.begin( restricPolSet[ polset_index ], 
         iterStrat, 0 );
       
       if( ( ! r1_it.end() ) && ( ! r2_it.end() ) )
       {
         r1_line = r1_it.currentLine();
         r1_col = r1_it.currentColumn();
         
         r2_line_off = r1_line - r2_it.currentLine();
         r2_col_off = r1_col - r2_it.currentColumn();         
  
         while( ! r1_it.end() ) {
           gotvalue1 = inputRaster1.getElement( r1_line, r1_col, 
             r1_value_double, inputRasterChn1 );
           gotvalue2 = inputRaster2.getElement( r1_line - r2_line_off, 
             r1_col - r2_col_off, r2_value_double, inputRasterChn2 );
           
           if( gotvalue1 )
           {
             r1_value_int = ( int ) r1_value_double;
             TEAGN_DEBUG_CONDITION( (r1_value_int + level_offset1) < 256,
               "Invalid histogram index");

             ++( init_r1hist[ r1_value_int + level_offset1 ] );
           }
           
           if( gotvalue2 )
           {
             r2_value_int = ( int ) r2_value_double;
             TEAGN_DEBUG_CONDITION( (r2_value_int + level_offset2) < 256,
               "Invalid histogram index");
           
             ++( init_r2hist[ r2_value_int + level_offset2 ] );
             
             if( gotvalue1 )
             {
               ++( init_jhist[ r1_value_int + level_offset1 ][ r2_value_int + 
                 level_offset2 ] );          
             }             
           }
           
           if( r1_line != last_r1_line ) {
             TEAGN_FALSE_OR_RETURN( progress.Increment(), 
               "Canceled by the user" );
             
             last_r1_line = r1_line;
           }
           
           ++r1_it;
           r1_line = r1_it.currentLine();
           r1_col = r1_it.currentColumn();           
         }
       }
     }  
   }    
  
   // locating the first and last non-null joint histogram position 
  
   unsigned int first_nonnul_jh_idx1 = 256;
   unsigned int first_nonnul_jh_idx2 = 256;
   unsigned int last_nonnul_jh_idx1 = 256;
   unsigned int last_nonnul_jh_idx2 = 256;
   {
     unsigned int hist_index1 = 0;
     unsigned int hist_index2 = 0;
     
     for( hist_index1 = 0 ; hist_index1 < 256 ; ++hist_index1 )
     {
       for( hist_index2 = 0 ; hist_index2 < 256 ; ++hist_index2 )
       {
         if( init_jhist[ hist_index1 ][ hist_index2 ] != 0 )
         {
           if( first_nonnul_jh_idx1 == 256 )
           {
             first_nonnul_jh_idx1 = last_nonnul_jh_idx1 = hist_index1;
           }
           else
           {
             last_nonnul_jh_idx1 = hist_index1;
           }
           
           if( first_nonnul_jh_idx2 == 256 )
           {
             first_nonnul_jh_idx2 = last_nonnul_jh_idx2 = hist_index2;
           }
           else
           {
             last_nonnul_jh_idx2 = hist_index2;
           }           
         }
       }
     }
   }   
   
   /* Building ordered joint histogram */
   
   if( first_nonnul_jh_idx1 != 256 )
   {
     unsigned int hist_index1 = 0;
     unsigned int hist_index2 = 0;
     std::pair< double, double > temp_pair;
     
     for( hist_index1 = first_nonnul_jh_idx2 ; hist_index1 <= last_nonnul_jh_idx1 ; 
       ++hist_index1 )
     {
       for( hist_index2 = first_nonnul_jh_idx2 ; hist_index2 <= last_nonnul_jh_idx2 ;
         ++hist_index2 )
       { 
         temp_pair.first = ( (double)hist_index1 ) - ( (double)level_offset1 );
         temp_pair.second = ( (double)hist_index2 ) - ( (double)level_offset2 );
           
         jHistContainer_[ temp_pair ] = init_jhist[ hist_index1 ][ hist_index2 ];
       }
     }
   }
      
   // locating the first and last non-null histograms position
  
   unsigned int first_nonnul_h_idx1 = 256;
   unsigned int first_nonnul_h_idx2 = 256;
   unsigned int last_nonnul_h_idx1 = 256;
   unsigned int last_nonnul_h_idx2 = 256;
   {
     for( unsigned int hist_index = 0 ; hist_index < 256 ; ++hist_index )
     {
       if( init_r1hist[ hist_index ] != 0 )
       {
         if( first_nonnul_h_idx1 == 256 )
         {
           first_nonnul_h_idx1 = last_nonnul_h_idx1 = hist_index;
         }
         else
         {
           last_nonnul_h_idx1 = hist_index;
         }
       }
       
       if( init_r2hist[ hist_index ] != 0 )
       {
         if( first_nonnul_h_idx2 == 256 )
         {
           first_nonnul_h_idx2 = last_nonnul_h_idx2 = hist_index;
         }
         else
         {
           last_nonnul_h_idx2 = hist_index;
         }
       }      
     }
   }   
   
   /* Building ordered raster 1 histogram */
   
   if( first_nonnul_h_idx1 != 256 )
   {
     unsigned int hist_index = 0;
     
     for( hist_index = first_nonnul_h_idx1 ; hist_index <= last_nonnul_h_idx1 ; 
       ++hist_index )
     {
       r1Histogram_[ ( (double)hist_index ) - ( (double)level_offset1 ) ] =
         init_r1hist[ hist_index ];
     }
   }   
   
   /* Building ordered raster 2 histogram */
   
   if( first_nonnul_h_idx2 != 256 )
   {
     unsigned int hist_index = 0;
     
     for( hist_index = first_nonnul_h_idx2 ; hist_index <= last_nonnul_h_idx2 ; 
       ++hist_index )
     {
       r2Histogram_[ ( (double)hist_index ) - ( (double)level_offset2 ) ] =
         init_r2hist[ hist_index ];
     }
   }    
   
   return true;
}

unsigned long int TePDIJointHistogram::getProgressSteps( 
  const TePolygonSet& polset, double resy )
{
  unsigned long int steps = 0;

  for( unsigned int ps_index = 0 ; ps_index < polset.size() ; 
    ++ps_index ) {
    
    const TeBox& polbox( polset[ ps_index ].box() );
    steps += ( unsigned long int)ABS( ( polbox.y2() - polbox.y1() ) / resy );
  }
  
  return steps;
}

