/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2007 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/
#include "TeRasterMemManager.h"

#include "TeUtils.h"
#include "TeErrorLog.h"
#include "TeAgnostic.h"


TeRasterMemManager::TeRasterMemManager()
{
  init();
}


TeRasterMemManager::~TeRasterMemManager()
{
  clear();
}


void TeRasterMemManager::clear()
{
  if( bands_nmb_ )
  {
    /* Cleaning the ram tiles ( the disk tile included !! ) */
    
    const unsigned int tottiles = (unsigned int)all_tiles_ptrs_vec_.size();
    
    for( unsigned int idx = 0 ; idx < tottiles ; ++idx )
    {
      if( all_tiles_ptrs_vec_[ idx ] != cur_disk_tile_ptr_ )
      {
        delete[] ( (TilePtrT)(all_tiles_ptrs_vec_[ idx ]) );
      }
    }
    
    if( cur_disk_tile_ptr_ ) delete[] cur_disk_tile_ptr_;
    
    /* closing and removing disk files */
    
    for( unsigned int odfidx = 0 ; odfidx < open_disk_files_vec_.size() ; 
      ++odfidx )
    {
      fclose( open_disk_files_vec_[ odfidx ].first );
      remove( open_disk_files_vec_[ odfidx ].second.c_str() );
    }
    
    /* cleaning containers */
    
    bands_tiles_sizes_.clear();
    all_tiles_ptrs_vec_.clear();
    tile2file_vec_.clear();
    open_disk_files_vec_.clear();

    /* Reseting the the default values */
    
    init();
  }
}

bool TeRasterMemManager::reset( unsigned int bands, 
  unsigned int tiles_per_band, const std::vector< unsigned int >& tiles_sizes,
  MemoryPolicy mem_pol )
{
  clear();
  
  TEAGN_TRUE_OR_RETURN( ( ( bands != 0 ) && ( tiles_per_band != 0 ) &&
    ( tiles_sizes.size() == bands ) ), "Invalid parameters" );

  tiles_per_band_ = tiles_per_band;
  bands_nmb_ = bands;
  bands_tiles_sizes_ = tiles_sizes;
  
  const unsigned int total_tiles_number = tiles_per_band_ * bands_nmb_;
  
  /* Allocating support vectors */
  
  try
  {
    all_tiles_ptrs_vec_.resize( tiles_per_band_ * bands_nmb_ );
    tile2file_vec_.resize( tiles_per_band_ * bands_nmb_ );
    
    Tiles2FileData dummydata;
    
    for( unsigned int tiles_vects_index = 0 ; 
      tiles_vects_index < total_tiles_number ;
      ++tiles_vects_index ) {
        
      all_tiles_ptrs_vec_[ tiles_vects_index ] = 0;
      tile2file_vec_[ tiles_vects_index ] = dummydata;
    }      
  }
  catch(...)
  {
    clear();
    
    TEAGN_LOG_AND_RETURN( 
      "Unable to allocate support vectors" )
  }
  
  /* Allocating the disk tiles mem buffer */
  
  try
  {
    unsigned int maxtilesize = 0;
    for( unsigned int curr_band = 0 ; 
      curr_band < bands_nmb_ ;  ++curr_band ) {    
      
      if( bands_tiles_sizes_[ curr_band ] > maxtilesize )
      {
        maxtilesize = bands_tiles_sizes_[ curr_band ];
      }
    }
    
    cur_disk_tile_ptr_ = (TilePtrT) new TileDataT[ maxtilesize ];
  }
  catch(...)
  {
    clear();
    
    TEAGN_LOG_AND_RETURN( 
      "Unable to allocate disk tiles RAM buffer" )
  }  
  
  /* Allocating tiles */
  
  if( mem_pol == DiskMemPol ) {
    if( ! allocateDiskTilesFiles( 0, 0 ) )
    {
      clear();
      
      return false;
    }
    else
    {
      TEAGN_TRUE_OR_THROW( getTilePointer( 0, 0 ),
        "Invalid tile pointer" )
    }
  } else {
    unsigned int curr_band = 0;
    unsigned int curr_tile = 0;
    TilePtrT newtile_ptr = 0;
    const unsigned long int max_ram_tiles = getMaxRAMTiles( bands_nmb_,
      tiles_per_band_, bands_tiles_sizes_ );
    unsigned int curr_ram_tiles_nmb = 0; 
    
    for( curr_band = 0 ; curr_band < bands ; ++curr_band ) {
      unsigned int tile_bytes = bands_tiles_sizes_[ curr_band ];
      TEAGN_TRUE_OR_THROW( ( tile_bytes > 0 ), "Invalid tile bytes" );      
      
      unsigned long int curr_tile_index = 0;
      
      for( curr_tile = 0 ; curr_tile < tiles_per_band_ ; ++curr_tile ) {
        curr_tile_index = ( curr_band * tiles_per_band_ ) + curr_tile;
        
        switch( mem_pol ) {
          case RAMMemPol :
          {
            try
            {
              newtile_ptr = (TilePtrT) new TileDataT[ tile_bytes ];
                
              if( newtile_ptr == 0 ) {
                clear();
                
                return false;
              } else {
                all_tiles_ptrs_vec_[ curr_tile_index ] = 
                  newtile_ptr;
              }
            }
            catch(...)
            {
              clear();
              
              return false;
            }
              
            break;
          }
          case AutoMemPol :
          {
            if( curr_ram_tiles_nmb <= max_ram_tiles ) {
              try
              {
                newtile_ptr = (TilePtrT) new TileDataT[ tile_bytes ];
                  
                if( newtile_ptr == 0 ) {
                  if( ! allocateDiskTilesFiles( curr_band, curr_tile ) )
                  {
                    clear();
                    
                    return false;
                  }
                  else
                  {
                    TEAGN_TRUE_OR_THROW( getTilePointer( curr_band, 
                      curr_tile ), "Invalid tile pointer" )
                      
                    /* break the tiles loop */
                    
                    curr_band = bands;
                    curr_tile = tiles_per_band_;
                  }
                } else {
                  all_tiles_ptrs_vec_[ curr_tile_index ] = 
                    newtile_ptr;
                  
                  ++curr_ram_tiles_nmb;
                }
              }
              catch(...)
              {
                if( ! allocateDiskTilesFiles( curr_band, curr_tile ) )
                {
                  clear();
                  
                  return false;
                }
                else
                {
                  TEAGN_TRUE_OR_THROW( getTilePointer( curr_band, 
                    curr_tile ), "Invalid tile pointer" )
                    
                  /* break the tiles loop */
                  
                  curr_band = bands;
                  curr_tile = tiles_per_band_;                    
                }
              }
            } else {
              if( ! allocateDiskTilesFiles( curr_band, curr_tile ) )
              {
                clear();
                
                return false;
              }
              else
              {
                TEAGN_TRUE_OR_THROW( getTilePointer( curr_band, 
                  curr_tile ), "Invalid tile pointer" )
                  
                /* break the tiles loop */
                
                curr_band = bands;
                curr_tile = tiles_per_band_;                  
              }              
            }
                            
            break;
          }
          default :
          {
            clear();
            
            TEAGN_LOG_AND_THROW( "Invalid memory policy" );
      
            break;
          }
        }
      }
    }
  }
  
  return true;
}


void TeRasterMemManager::init()
{
  tiles_per_band_ = 0;
  bands_nmb_ = 0;
  max_disk_file_size_ = 1024 * 1024 * 100;
  cur_disk_tile_ptr_ = 0;
  cur_disk_tile_idx_ = -1;
}


bool TeRasterMemManager::allocateDiskTilesFiles( 
  unsigned int starting_band_index, unsigned int starting_tile_index )
{
  TEAGN_TRUE_OR_THROW( ( bands_tiles_sizes_.size() == bands_nmb_ ),
    "Invalid tile sizes vector" );    
  TEAGN_TRUE_OR_THROW( ( starting_band_index < bands_nmb_ ),
    "Invalid starting_band_index" );    
  TEAGN_TRUE_OR_THROW( ( starting_tile_index < tiles_per_band_ ),
    "Invalid starting_tile_index" );    
 
  /* Allocating disk files */    
  
  for( unsigned int curr_band = starting_band_index ; curr_band < bands_nmb_ ; 
    ++curr_band ) {
  
    const unsigned int tile_size = bands_tiles_sizes_[ curr_band ];
    TEAGN_TRUE_OR_THROW( ( tile_size <= max_disk_file_size_ ), 
      "Invalid tile size" );
    
    const unsigned long int tiles_per_file = ( unsigned long int )
      floor( ( (double)max_disk_file_size_ ) / ( (double) tile_size ) );
      
    const unsigned long int file_size = (unsigned long int)
      ( tiles_per_file * tile_size );
    
    unsigned int curr_tiles_in_file = tiles_per_file + 1;
      
    unsigned int curr_tile = 0;
    if( curr_band == starting_band_index ) {
      curr_tile = starting_tile_index;
    }
    
    unsigned long int curr_tile_index = 0;
    
    FILE* currfileptr = 0;
    std::string currfilename;
      
    while( curr_tile < tiles_per_band_ ) {
      curr_tile_index = ( curr_band * tiles_per_band_ ) + curr_tile;
    
      if( curr_tiles_in_file >= tiles_per_file ) {
        /* create a new disk file */

        if( ! createNewDiskFile( file_size, currfilename, &currfileptr ) )
        {
          TEAGN_LOGERR( "Unable to create temporary disk file" );
          
          clear();
          
          return false;           
        }
        
        std::pair< FILE*, std::string > temppair;
        temppair.first = currfileptr;
        temppair.second = currfilename;
        
        open_disk_files_vec_.push_back( temppair );
        
        curr_tiles_in_file = 0;
      }
      
      tile2file_vec_[ curr_tile_index ].fileptr_ = currfileptr;
      tile2file_vec_[ curr_tile_index ].fileoff_ = curr_tiles_in_file *
        tile_size;
      tile2file_vec_[ curr_tile_index ].tilesize_ = tile_size;
      
      ++curr_tiles_in_file;
      
      ++curr_tile;
    }
    
  }
  
  return true;
}


unsigned long int TeRasterMemManager::getMaxRAMTiles( unsigned int bands, 
  unsigned int tiles_per_band,
  const std::vector< unsigned int >& tiles_sizes)
{
  const unsigned long int free_vm = TeGetFreeVirtualMemory();
  
  if( free_vm < max_disk_file_size_ )
  {
    return 0;
  }
  else
  {
    const unsigned int max_ram  = (unsigned int)
      ( 0.90 * ( (double)( free_vm ) ) );    
    unsigned long int max_ram_tiles = 0;
    unsigned long int curr_used_ram = 0;
    
    for( unsigned int curr_band = 0 ; curr_band < bands ; 
      ++curr_band ) {
    
      const unsigned int tile_size = tiles_sizes[ curr_band ];
      
      for( unsigned int curr_tile = 0 ; curr_tile < tiles_per_band ;
        ++curr_tile ) {
        
        if( ( curr_used_ram + tile_size ) > max_ram ) {
          curr_tile = tiles_per_band;
          curr_band = bands;
        } else {
          ++max_ram_tiles;
          curr_used_ram += tile_size;
        }
      }
    }

    return max_ram_tiles;
  }
}


void* TeRasterMemManager::getTilePointer( const unsigned int& band, 
  const unsigned int& tile)
{
  const unsigned int reqtileindex = ( band * tiles_per_band_ ) + tile;
  
  TEAGN_DEBUG_CONDITION( ( reqtileindex < ( tiles_per_band_ * bands_nmb_ ) ),
    "The required band/tile out of range" );
    
  TilePtrT tileptr = all_tiles_ptrs_vec_[ reqtileindex ];
  
  if( tileptr == 0 ) {
    /* Flushing the current disk tile back to disk */
  
    if( cur_disk_tile_idx_ > (-1) ) {
      Tiles2FileData& curtiledata = tile2file_vec_[
        cur_disk_tile_idx_ ];
      
      TEAGN_TRUE_OR_THROW( 0 == fseek( curtiledata.fileptr_, 
        (long)( curtiledata.fileoff_ ), SEEK_SET ),
        "File seek error" );
        
      TEAGN_TRUE_OR_THROW( 1 == fwrite( (void*)cur_disk_tile_ptr_, 
        (size_t)( curtiledata.tilesize_ ), 1, curtiledata.fileptr_ ),
        "File write error" )        
        
      all_tiles_ptrs_vec_[ cur_disk_tile_idx_ ] = 0;
    }
    
    /* Reading the required tile into RAM */
    
    Tiles2FileData& reqtiledata = tile2file_vec_[
      reqtileindex ];    
    
    TEAGN_TRUE_OR_THROW( 0 == fseek( reqtiledata.fileptr_, 
      (long)( reqtiledata.fileoff_ ), SEEK_SET ),
      "File seek error" )
      
    TEAGN_TRUE_OR_THROW( 1 == fread( (void*)cur_disk_tile_ptr_, 
      (size_t)( reqtiledata.tilesize_ ), 1, reqtiledata.fileptr_ ),
      "File write error" )
      
    all_tiles_ptrs_vec_[ reqtileindex ] = cur_disk_tile_ptr_;
      
    cur_disk_tile_idx_ = reqtileindex;
      
    return cur_disk_tile_ptr_;
  }
  else
  {
    return tileptr;
  }
}


bool TeRasterMemManager::createNewDiskFile( unsigned long int size,
  std::string& filename, FILE** fileptr )
{
  TEAGN_TRUE_OR_RETURN( TeGetTempFileName( filename ),
    "Unable to get temporary file name" );
    
  (*fileptr) = fopen( filename.c_str(), "wb+" );
  TEAGN_TRUE_OR_RETURN( (*fileptr) != 0, "Invalid file pointer" )
  
  long seekoff = (long)( size - 1 );
  
  if( 0 != fseek( (*fileptr), seekoff, SEEK_SET ) )
  {
    fclose( (*fileptr) );
    TEAGN_LOGERR( "File seek error" );
    return false;
  }

  unsigned char c = '\0';
  if( 1 != fwrite( &c, 1, 1, (*fileptr) ) )
  {
    fclose( (*fileptr) );
    TEAGN_LOGERR( "File write error" );
    return false;
  }
    
  return true;
}

