/*
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*/

#include "TePDIChainCode4.hpp"

//-----------------------------------------------------------------


static const int NACO_SIZE = 128;



//-----------------------------------------------------------------
bool
TePDIChainCode4 :: isOK()
{
        return Flag;
}

//-----------------------------------------------------------------
int
TePDIChainCode4 :: numDirections()
{
        return NDir;
}

//-----------------------------------------------------------------

TePDIChainCode4 :: TePDIChainCode4( int naco )
{
        initialize( naco );
}

//-----------------------------------------------------------------

TePDIChainCode4 :: TePDIChainCode4()
{
        initialize( NACO_SIZE );
}

//-----------------------------------------------------------------

TePDIChainCode4 :: TePDIChainCode4( const TePDIChainCode4 &cha )
{
        *this = cha;
}

//-----------------------------------------------------------------

TePDIChainCode4 :: ~TePDIChainCode4()
{
        freeBuffer();
}

//-----------------------------------------------------------------

void
TePDIChainCode4 :: initialize( int naco )
{
        Flag = true;
        firstPoint.setXY(0,0);
        Rh   = 0.;
        Rv   = 0.;
        NDir = 0;
        NBytes = 0;
        if( naco <= 0 )
                Naco = NACO_SIZE;
        else
                Naco = naco;
        Cadeia = 0;
        if( allocBuffer( Naco ) == 0 )
                Flag = false;
}

//-----------------------------------------------------------------

TePDIChainCode4&
TePDIChainCode4 :: operator=( const TePDIChainCode4 &cha )
{
        if( cha.Flag == false ) return *this;
        if( Flag == false ) return *this;

        Flag = cha.Flag;
        firstPoint    = cha.firstPoint;
        Rh   = cha.Rh;
        Rv   = cha.Rv;
        NDir = cha.NDir;

        freeBuffer();

        if( allocBuffer( cha.NBytes ) == 0 ){
                Flag = false;
        }
        else{
                Flag = true;
                for( int i = 0; i < NBytes; i++ ) Cadeia[i] = cha.Cadeia[i];
        }

        return *this;
}

//-----------------------------------------------------------------

bool
TePDIChainCode4 :: allocBuffer( int tamanho )
{
        Cadeia = new char [ tamanho ];
        NBytes = tamanho;
        return ( Cadeia == 0 ) ? 0 : 1;
}

//-----------------------------------------------------------------

void
TePDIChainCode4 :: freeBuffer()
{
        if( Cadeia == 0 )
                return;
        delete [] Cadeia;
        Cadeia = 0;
        NBytes = 0;
        return;
}

//-----------------------------------------------------------------

bool
TePDIChainCode4 :: expandBuffer()
{
        int     tamanho = NBytes+Naco;
        char *cadeia1 = new char [ tamanho ];
        if( cadeia1 == 0 )
                return false;

        for( int i = 0; i < NBytes; i++ ){
                cadeia1[i] = Cadeia[i];
        }

        freeBuffer();

        Cadeia = cadeia1;
        NBytes = tamanho;

        return true;
}


//-----------------------------------------------------------------

bool
TePDIChainCode4 :: setFirstPoint( TeCoord2D P)  //FAMI0695
{
        firstPoint=P;
        NDir = 0;       // <<< no data

        return true;
}

//-----------------------------------------------------------------

void
TePDIChainCode4 :: getFirstPoint( TeCoord2D &P ) const  //FAMI0695
{
        P=firstPoint;
        return;
}

//-----------------------------------------------------------------

bool
TePDIChainCode4 :: setResolution( double rh, double rv )
{
        Rh = rh;
        Rv = rv;
        return true;
}

//-----------------------------------------------------------------

void
TePDIChainCode4 :: getResolution( double *rh, double *rv ) const
{
        *rh = Rh;
        *rv = Rv;
}

//-----------------------------------------------------------------

bool
TePDIChainCode4 :: add( int dir )
{
        if( Flag == false )
                return false;

        if( NDir >= NBytes * 4 ){
                if( expandBuffer() == 0 ){
                        Flag = false;
                        return false;
                }
        }

        Cadeia[ NDir/4 ] = ( (unsigned int)( Cadeia[ NDir/4 ] )
                                &
                            ~( 0x03 << ( ( NDir % 4) * 2 ) )
                           )
                                |
                           ( (unsigned int)dir << ( ( NDir % 4 ) * 2 ) ) ;
        NDir++;

        //DEP return c->n;

        return true;
}

//-----------------------------------------------------------------

bool
TePDIChainCode4 :: getDirection( int pos, int &dir ) const
{
        if( Flag == false )
                return false;

        dir = ( ((unsigned int)(Cadeia[pos/4]) ) >> ( ( pos % 4 ) * 2 ) )  & 0x03 ;
        return true;
}

//-----------------------------------------------------------------

