/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIHARALICK_HPP
  #define TEPDIHARALICK_HPP

  #include "TePDIAlgorithm.hpp"
  #include "TePDIParameters.hpp"
  #include "TePDITypes.hpp"
  
  #include <TeSharedPtr.h>
  
  #include <map>

  /**
   * @brief This is the class for Haralick texture features over raster data.
   * @author Vanessa Oliveira de Souza <vanessa@dpi.inpe.br>
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   * @ingroup TePDIGeneralAlgoGroup
   *
   * @note The general required parameters are:
   *
   * @param input_raster (TePDITypes::TePDIRasterPtrType) Input image.
   * @param polygonset ( TePDITypes::TePDIPolygonSetPtrType ) - Restriction 
   * polygon set - The area where to do calcules over all rasters.
   */
  class PDI_DLL TePDIHaralick : public TePDIAlgorithm {
    public :
    
      /** @typedef TeSharedPtr< TePDIHaralick > pointer 
          Type definition for a instance pointer */
      typedef TeSharedPtr< TePDIHaralick > pointer;
      
      /** @typedef const TeSharedPtr< TePDIHaralick > const_pointer 
          Type definition for a const instance pointer */
      typedef const TeSharedPtr< TePDIHaralick > const_pointer;   
      
      /** @enum DirectionT Concurrence matrix used directions */ 
      enum DirectionT {
        /** @brief North direction */
        North = 1,
        /** @brief NorthEast direction */
        NorthEast = 2,
        /** @brief East direction */
        East = 4,
        /** @brief SouthEast direction */
        SouthEast = 8,
        /** @brief South direction */
        South = 16,
        /** @brief SouthWest direction */
        SouthWest = 32,
        /** @brief West direction */
        West = 64,
        /** @brief NorthWest direction */
        NorthWest = 128
      };

      /**
       * @brief Default Constructor.
       * @param dirmask Directions mask( DirectionT );
       *
       */
      TePDIHaralick( int dirmask = East );

      /**
       * @brief Default Destructor
       */
      ~TePDIHaralick();

      /**
       * @brief Checks if the supplied parameters fits the requirements of each
       * PDI algorithm implementation.
       *
       * @note Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( const TePDIParameters& parameters ) const;
                
      /**
       * @brief Entropy of one polygon inside one band.
       *
       * @param band Raster band.
       * @param pol_index The polygon index inside the polygon set
       * ( if no polygon set is present, use de default zero value ).
       * @param value The generated value.
       * @return true if OK, false on errors.
       */
      bool getEntropy( unsigned int band, unsigned int pol_index,
        double& value );
        
       /**
        * @brief Energy of one polygon inside one band.
        * The energy measures textural uniformity, that is, repetition of pairs of pixel in the image. 
        * High values of energy indicate that the distribution of gray level in region in the image 
        * has a constant distribution or a periodic form.
        *
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param energyValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getEnergy( unsigned int band, unsigned int pol_index,
        double& energyValue );
  
        /**
        * @brief Contrast of one polygon inside one band.
        * The contrast is a estimate of the local variations. 
        * Average of the squares of the differences between pixels.
        *
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param contrastValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getContrast( unsigned int band, unsigned int pol_index,
        double& contrastValue );
  
        /**
        * @brief Homogeneity of one polygon inside one band.
        * This measure of texture results in great values for regions with similar gray level. 
        * The contrast and the homogeneity are inversely correlacionad
        *
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param homogeneityValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getHomogeneity( unsigned int band, unsigned int pol_index,
        double& homogeneityValue );
  
        /**
        * @brief QuiSquare of one polygon inside one band.
        * This measure can be understood as a normalization of the energy for the linear dependence 
        * of the gray level tones in a region in the image. 
        *        
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param QuiSquareValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getQuiSquare( unsigned int band, unsigned int pol_index,
        double& QuiSquareValue );        
  
    protected :
    
      /** @typedef std::pair< double, double > ConcMatrixKeyT
        * Concurrence matrix key type( A coord pair where x=first pixel value 
        * and y=second pixel value).
        */    
      typedef std::pair< double, double > ConcMatrixKeyT;
    
      /** @typedef std::map< ConcMatrixKeyT, double > ConcMatrixT
        * Concurrence matrix type. 
        * @param ConcMatrixKeyT A coord pair where x=first pixel value and 
        * y=second pixel value.
        * @param double The pair frequency ( The frequency field must be 
        * double type due the matrix normalizing ).
        */
      typedef std::map< ConcMatrixKeyT, double > ConcMatrixT;
      
      /** @typedef TeSharedPtr< ConcMatrixT > ConcMatrixPtrT
        * Concurrence matrix pointer type. 
        */
      typedef TeSharedPtr< ConcMatrixT > ConcMatrixPtrT;
      
      /** @typedef std::pair< unsigned int, unsigned int > ConcMatrixCacheKeyT
        * Concurrence matrix cache key type.
        */
      typedef std::pair< unsigned int, unsigned int > ConcMatrixCacheKeyT;
      
      /** @typedef std::map< ConcMatrixCacheKeyT, ConcMatrixPtrT > ConcMatrixCacheT
        * Concurrence matrix cache type.
        */
      typedef std::map< ConcMatrixCacheKeyT, ConcMatrixPtrT > ConcMatrixCacheT;    
 
      /** @brief Direction mask. */
      int dirmask_;
      
      /** @brief Concurrent matrix cache. */
      ConcMatrixCacheT conc_matrix_cache_;
      
      /** @brief Input raster pointer. */
      TePDITypes::TePDIRasterPtrType input_raster_;
      
      /** @brief Polygon set pointer. */
      TePDITypes::TePDIPolygonSetPtrType polygonset_;
      
      /**
       * @brief Runs the current algorithm implementation.
       *
       * @return true if OK. false on error.
       */
      bool RunImplementation();      
      
      /**
       * @brief Reset the internal state to the initial state.
       *
       * @param params The new parameters referente at initial state.
       */
      void ResetState( const TePDIParameters& params );      
      
      /**
       * @brief Build a concurrence matrix using a supplied polygon.
       *
       * @param band Raster band.
       * @param pol_index The polygon index inside the polygon set
       * ( if no polygon set is present, use de default zero value ).
       * @return A pointer to the generated matrix or a inactive pointer
       * on errors.
       */      
      ConcMatrixPtrT getConcurrenceMatrix( unsigned int band, 
        unsigned int pol_index );
   
  };
  
/** @example TePDIHaralick_test.cpp
 *    Shows how to use this class.
 */    

#endif
